#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmProcessShipmentIn;
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_BloodUnit : BaseTest 
	{
		private DataTable _dtShipment;
		private DataTable _dtPatient;
		private DataTable _dtBloodUnitCodabar;
		private DataTable _dtBloodUnitISBT;
		private DataTable _dtLocalSupplier;
		
		private BOL.BloodUnit _tstCodabar;
		private Guid _codabarGuid;
		private BOL.BloodUnit _tstISBT;
		private Guid _isbtGuid;
        private BOL.LockManager _lockManager;
		
		private Guid _cannedCommentGuid = Guid.Empty;

		[SetUp]
		protected void SetUp() 
		{
			this.BaseSetUp();

            this._lockManager = new BOL.LockManager(-35407, -35407, BOL.LockManager.DefaultLockInactivityTimeoutMin);
            this._lockManager.LockFunction = COMMON.LockFunctions.UnitTests;

			if (this.RefreshData)
			{

				this._dtPatient = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Patient", true);

				//Codabar Unit
				this._dtBloodUnitCodabar = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE LabelTypeCode = 'C' AND IncomingShipmentGuid IS NOT NULL AND RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND BloodUnitGuid NOT IN (SELECT ToBloodUnitGuid FROM BloodUnitModification)" , false);
				if (this._dtBloodUnitCodabar.Rows.Count == 0)	DataCreator.CreateBloodUnit(true);
				this._dtBloodUnitCodabar = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE LabelTypeCode = 'C' AND IncomingShipmentGuid IS NOT NULL AND RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND BloodUnitGuid NOT IN (SELECT ToBloodUnitGuid FROM BloodUnitModification)" , true);

				//ISBT Unit
				this._dtBloodUnitISBT = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE LabelTypeCode = 'I' AND IncomingShipmentGuid IS NOT NULL AND RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND BloodUnitGuid NOT IN (SELECT ToBloodUnitGuid FROM BloodUnitModification)" , false);
				if (this._dtBloodUnitISBT.Rows.Count == 0)	DataCreator.CreateBloodUnit(false);
				this._dtBloodUnitISBT = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE LabelTypeCode = 'I' AND IncomingShipmentGuid IS NOT NULL AND RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND BloodUnitGuid NOT IN (SELECT ToBloodUnitGuid FROM BloodUnitModification)" , true);

				this._codabarGuid = (Guid)this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.BloodUnitGuid];
				this._isbtGuid = (Guid)this._dtBloodUnitISBT.Rows[0][TABLES.BloodUnit.BloodUnitGuid];

				//Local Supplier
				this._dtLocalSupplier = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM LocalSupplier WHERE RecordStatusCode <> 'X' AND FacilityPrefix <> '' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'" , false);
				if (this._dtLocalSupplier.Rows.Count == 0)	DataCreator.CreateLocalSupplier();
				this._dtLocalSupplier = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM LocalSupplier WHERE RecordStatusCode <> 'X' AND FacilityPrefix <> '' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'" , true);
				

				//Shipment
				this._dtShipment = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM BloodUnitShipment WHERE IncomingShipmentGuid IS NOT NULL AND RecordStatusCode = 'A' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM BloodUnit WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "')", true);

				//CannedComment
				this._cannedCommentGuid = DataCreator.GetRecordGuid("SELECT TOP 1 CannedCommentGuid FROM CannedComment WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");

				this.RefreshData = false;
			}

			this._tstCodabar = new BOL.BloodUnit(this._codabarGuid);
			this._tstISBT = new BOL.BloodUnit(this._isbtGuid);
			
		}

		
		[Test]
		public void Ctor_String_BarcodeType_Guid_Pass()
		{
			//Test Codabar
			BOL.BloodUnit tstCodabar = new BOL.BloodUnit(this._tstCodabar.EyeReadableUnitId, this._tstCodabar.FullProductCode, this._tstCodabar.BarcodeType, this._tstCodabar.BloodProductGuid);

			Assert.AreEqual(this._tstCodabar.BloodUnitGuid, tstCodabar.BloodUnitGuid, "Codabar-BloodUnitGuid" );
			Assert.AreEqual(this._tstCodabar.BloodProductGuid, tstCodabar.BloodProductGuid, "Codabar-BloodProductGuid");
			Assert.AreEqual(this._tstCodabar.ProductCode, tstCodabar.ProductCode, "Codabar-ProductCode");
			Assert.AreEqual(this._tstCodabar.BarcodeType, tstCodabar.BarcodeType, "Codabar-BarcodeType");
			Assert.AreEqual(this._tstCodabar.EyeReadableUnitId, tstCodabar.EyeReadableUnitId, "Codabar-EyeReadableUnitId");
			Assert.AreEqual(false, tstCodabar.IsNew, "Codabar-IsNew");

			//Test ISBT
			BOL.BloodUnit tstISBT = new BOL.BloodUnit(this._tstISBT.EyeReadableUnitId, this._tstISBT.FullProductCode, this._tstISBT.BarcodeType, this._tstISBT.BloodProductGuid);

			Assert.AreEqual(this._tstISBT.BloodUnitGuid, tstISBT.BloodUnitGuid, "ISBT-BloodUnitGuid" );
			Assert.AreEqual(this._tstISBT.BloodProductGuid, tstISBT.BloodProductGuid, "ISBT-BloodProductGuid");
			Assert.AreEqual(this._tstISBT.ProductCode, tstISBT.ProductCode, "ISBT-ProductCode");
			Assert.AreEqual(this._tstISBT.BarcodeType, tstISBT.BarcodeType, "ISBT-BarcodeType");
			Assert.AreEqual(this._tstISBT.EyeReadableUnitId, tstISBT.EyeReadableUnitId, "ISBT-EyeReadableUnitId");
			Assert.AreEqual(false, tstISBT.IsNew, "ISBT-IsNew");
		}

		[Test]
		public void Ctor_String_BarcodeType_Guid_Fail()
		{
			//Test Codabar
			Guid bpCodabar = DataCreator.GetRecordGuid("SELECT BloodProductGuid FROM BloodProduct WHERE ProductCode = '04020'");
			string codabarUnitId = new Random().Next(11111,99999).ToString() + new Random().Next(11111,99999).ToString();


			BOL.BloodUnit tstCodabar = new BOL.BloodUnit(codabarUnitId, "04020", COMMON.BarcodeType.Codabar, bpCodabar);

			Assert.AreEqual("04020", tstCodabar.ProductCode, "Codabar-ProductCode");
			Assert.AreEqual(COMMON.BarcodeType.Codabar, tstCodabar.BarcodeType, "Codabar-BarcodeType");
			Assert.AreEqual(codabarUnitId, tstCodabar.EyeReadableUnitId, "Codabar-EyeReadableUnitId");
			Assert.AreEqual(true, tstCodabar.IsNew, "Codabar-IsNew");

			//Test ISBT
			Guid bpISBT = DataCreator.GetRecordGuid("SELECT BloodProductGuid FROM BloodProduct WHERE ProductCode = 'E0001'");
			
			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 EyeReadableUnitId FROM BloodUnit WHERE LabelTypeCode = 'I' AND UnitProductCode <> 'E0001'");
			if (dtTemp.Rows.Count == 0)
			{
				DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, DataCreator.GetRandomBloodProduct(false, COMMON.ComponentClass.CRYO).BloodProductGuid, false);
				dtTemp = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 EyeReadableUnitId FROM BloodUnit WHERE LabelTypeCode = 'I' AND UnitProductCode <> 'E0001'");
			}
			string isbtUnitId = dtTemp.Rows[0][0].ToString();

			BOL.BloodUnit tstISBT = new BOL.BloodUnit(isbtUnitId, "E0001V00", COMMON.BarcodeType.ISBT128, bpISBT);
			//TODO: Investigate flaw in BLOODUNIT constructor - Passing in as above - Resets the productCode
			//		in the LoadFromDataRow() method, around line 250ish
			tstISBT.ProductCode = "E0001";
			Assert.AreEqual("E0001", tstISBT.ProductCode, "ISBT-ProductCode");
			Assert.AreEqual(COMMON.BarcodeType.ISBT128, tstISBT.BarcodeType, "ISBT-BarcodeType");
			Assert.AreEqual(isbtUnitId, tstISBT.EyeReadableUnitId, "ISBT-EyeReadableUnitId");
			Assert.AreEqual(true, tstISBT.IsNew, "ISBT-IsNew");
		}

		[Test]
		public void Ctor_Guid_Pass()
		{
			//Test Codabar
			BOL.BloodUnit tstCodabar = new BOL.BloodUnit(this._codabarGuid);
			
			Assert.AreEqual(this._codabarGuid, tstCodabar.BloodUnitGuid, "Codabar-BloodUnitGuid");
			Assert.AreEqual(false, tstCodabar.IsNew, "Codabar-IsNew");

			//Test ISBT
			BOL.BloodUnit tstISBT = new BOL.BloodUnit(this._isbtGuid);
			
			Assert.AreEqual(this._isbtGuid, tstISBT.BloodUnitGuid, "ISBT-BloodUnitGuid");
			Assert.AreEqual(false, tstISBT.IsNew, "ISBT-IsNew");
		}

		[Test]
		public void Ctor_Guid_Fail()
		{
			//Test new Guid
			Guid unitGuid = Guid.NewGuid();
			BOL.BloodUnit tstBU = new BOL.BloodUnit(unitGuid);
			
			Assert.IsNotNull(tstBU, "Object");
			Assert.AreEqual(unitGuid, tstBU.BloodUnitGuid, "BloodUnitGuid");
			Assert.AreEqual(true, tstBU.IsNew, "IsNew");
		}

		[Test]
		public void Ctor_Guid_Guid_Pass()
		{
			//Test Codabar
			Guid bpCodabar = (Guid)this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.BloodProductGuid];

			BOL.BloodUnit tstCodabar = new BOL.BloodUnit(this._codabarGuid, bpCodabar);
			
			Assert.AreEqual(this._codabarGuid, tstCodabar.BloodUnitGuid, "Codabar-BloodUnitGuid");
			Assert.AreEqual(bpCodabar, tstCodabar.BloodProductGuid, "Codabar-BloodProductGuid");
			Assert.AreEqual(false, tstCodabar.IsNew, "Codabar-IsNew");

			//Test ISBT
			Guid bpISBT = (Guid)this._dtBloodUnitISBT.Rows[0][TABLES.BloodUnit.BloodProductGuid];

			BOL.BloodUnit tstISBT = new BOL.BloodUnit(this._isbtGuid, bpISBT);
			
			Assert.AreEqual(this._isbtGuid, tstISBT.BloodUnitGuid, "BloodUnitGuid");
			Assert.AreEqual(bpISBT, tstISBT.BloodProductGuid, "BloodProductGuid");
			Assert.AreEqual(false, tstISBT.IsNew, "IsNew");
		}

		[Test]
		public void Ctor_Guid_Guid_Fail()
		{
			//Test new Guids
			Guid unitGuid = Guid.NewGuid();
			Guid productGuid =  Guid.NewGuid();

			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid, productGuid);
			
			Assert.IsNotNull(tmpBU, "Object");
			Assert.AreEqual(unitGuid, tmpBU.BloodUnitGuid, "BloodUnitGuid");
			Assert.AreEqual(true, tmpBU.IsNew, "IsNew");
		}

		[Test]
		public void Ctor_DataRow_Pass()
		{
			//Test Codabar
			BOL.BloodUnit tstCodabar = new BOL.BloodUnit(this._dtBloodUnitCodabar.Rows[0]);
			Assert.AreEqual(this._codabarGuid, tstCodabar.BloodUnitGuid, "Codabar-BloodUnitGuid");

			//Test Codabar
			BOL.BloodUnit tstISBT = new BOL.BloodUnit(this._dtBloodUnitISBT.Rows[0]);
			Assert.AreEqual(this._isbtGuid, tstISBT.BloodUnitGuid, "ISTBT-BloodUnitGuid");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void Ctor_DataRow_Fail()
		{
			//Invalid DataRow
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(this._dtPatient.NewRow());
		}

		[Test]
		public void IncomingShipment_Pass()
		{
			BOL.IncomingShipment tmpIS = new BOL.IncomingShipment(this._tstCodabar.IncomingShipmentGuid);

			this._tstCodabar.IncomingShipment = tmpIS;

			Assert.AreEqual(tmpIS, this._tstCodabar.IncomingShipment, "IncomingShipment");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void IncomingShipment_Fail()
		{
			this._tstCodabar.IncomingShipment = new BOL.IncomingShipment(Guid.NewGuid());

		}

		// CR2899: This ElgibleForEmergencyIssue method changed in order to make processing more
		// efficient.  Instead of it processing the specimen, the calling method must now to it
		// and pass in booleans that reflect the results.
		//
		// As a result of this change, the corresponding unit tests also changed.
		[Test]
		public void ElgibleForEmergencyIssue_Pass()
		{
			// Developer notes: optional XM is irrelevant for all but the type of Other.


			#region RBC 
		
			// Only type O with untested specimen
			// No ABO/Rh discrepancy
			this._tstCodabar.BloodComponentClassID = (int)COMMON.ComponentClass.RBC;



			#region type O, no discrepancy, TAS tested

			//RBC, O Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;

			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			bool retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, true);
			Assert.AreEqual(true, retValue, "RBC");


			//RBC, O Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;

			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, true);
			Assert.AreEqual(true, retValue, "RBC");

			#endregion



			#region discrepancies and TAS untested

			//RBC, O Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;

			// Scenario: no optional XM, discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, false);
			Assert.AreEqual(true, retValue, "RBC");


			//RBC, O Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;

			// Scenario: no optional XM, no discrepancy, no TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "RBC");

			
			//RBC, O Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;

			// Scenario: no optional XM, discrepancy, no TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, false);
			Assert.AreEqual(true, retValue, "RBC");



			//RBC, O Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;

			// Scenario: no optional XM, discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, false);
			Assert.AreEqual(true, retValue, "RBC");


			//RBC, O Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;

			// Scenario: no optional XM, no discrepancy, no TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "RBC");

			
			//RBC, O Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;

			// Scenario: no optional XM, discrepancy, no TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, false);
			Assert.AreEqual(true, retValue, "RBC");

			#endregion



			#region type specific
			
			// Type specific are fine with valid ABO/Rh test and no discrepancy

			//RBC, A Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.A;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;

			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "RBC");


			//RBC, B Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.B;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;

			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "RBC");


			//RBC, AB Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;

			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
		        Assert.AreEqual(true, retValue, "RBC");



			//RBC, A Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.A;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;

			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "RBC");


			//RBC, B Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.B;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;

			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "RBC");


			//RBC, AB Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;

			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "RBC");

			#endregion


			#endregion
			


			#region FFP

			// Only type AB with untested specimen and no ABO/Rh discrepancy

			this._tstCodabar.BloodComponentClassID = (int)COMMON.ComponentClass.FFP;



			#region no discrepancy, ABO/Rh tested

			//FFP, AB Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
		
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
		        Assert.AreEqual(true, retValue, "FFP");


			//FFP, AB Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
	
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");


			// CR2567
			//FFP, AB
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Blank;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");


			//FFP, AB Rh inconclusive
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Inconclusive;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");


			//FFP, AB Rh not provided
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.NotProvided;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");

			// CR2567 end

			#endregion



			#region discrepancies and TAS untested

			//FFP, AB Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
		
			// Scenario: no optional XM, discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, true);
			Assert.AreEqual(true, retValue, "FFP");


			//FFP, AB Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
		
			// Scenario: no optional XM, no discrepancy, TAS untested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, false);
			Assert.AreEqual(true, retValue, "FFP");



			//FFP, AB Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
		
			// Scenario: no optional XM, discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, true);
			Assert.AreEqual(true, retValue, "FFP");


			//FFP, AB Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
		
			// Scenario: no optional XM, no discrepancy, TAS untested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, false);
			Assert.AreEqual(true, retValue, "FFP");

			#endregion



			#region type specific
			
			//FFP, A Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.A;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
		
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");


			//FFP, B Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.B;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
		
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");


			//FFP, O Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
		
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");



			//FFP, A Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.A;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
		
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");


			//FFP, B Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.B;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
		
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");


			//FFP, O Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
		
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "FFP");


			#endregion




			#endregion



			#region platelet

			// All are eligible

			//PLT
			this._tstCodabar.BloodComponentClassID = (int)COMMON.ComponentClass.PLT;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "PLT");


			// Worst case scenario: no optional XM, discrepancy, TAS untested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, false);
			Assert.AreEqual(true, retValue, "PLT");

			#endregion



			#region Cryo

			// All are eligible

			//CRYO
			this._tstCodabar.BloodComponentClassID = (int)COMMON.ComponentClass.CRYO;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "CRYO");

			// Worst case scenario: no optional XM, discrepancy, TAS untested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, true, false);
			Assert.AreEqual(true, retValue, "CRYO");

			#endregion


			#region Other

			// All are eligible if no optional XM ordered
			
			
			#region no optional XM ordered

			//OTHER
			this._tstCodabar.BloodComponentClassID = (int)COMMON.ComponentClass.Other;


			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "Other1");


			//OTHER, O Neg
			this._tstCodabar.ProductType.ProductTypeCode = "E034";
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "Other2");



			//OTHER, AB Pos
			this._tstCodabar.ProductType.ProductTypeCode = "E034";
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, true);
			Assert.AreEqual(true, retValue, "Other2");

			#endregion



			#region optional XM ordered, type O

			//OTHER, O Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(true, false, true);
			Assert.AreEqual(true, retValue, "Other2");


			//OTHER, O Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.O;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(true, false, true);
			Assert.AreEqual(true, retValue, "Other2");

			#endregion



			#region optional XM ordered, type specific

			//OTHER, A Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.A;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(true, false, true);
			Assert.AreEqual(true, retValue, "Other2");


			//OTHER, B Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.B;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(true, false, true);
			Assert.AreEqual(true, retValue, "Other2");


			//OTHER, AB Pos
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Positive;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(true, false, true);
			Assert.AreEqual(true, retValue, "Other2");


			//OTHER, A Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.A;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(true, false, true);
			Assert.AreEqual(true, retValue, "Other2");


			//OTHER, B Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.B;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(true, false, true);
			Assert.AreEqual(true, retValue, "Other2");


			//OTHER, AB Neg
			this._tstCodabar.BloodUnitMedia.AboRh.Abo = COMMON.ABO.AB;
			this._tstCodabar.BloodUnitMedia.AboRh.RH = COMMON.RH.Negative;
			
			// Happy day scenario: no optional XM, no discrepancy, TAS tested
			retValue = this._tstCodabar.EligibleForEmergencyIssue(true, false, true);
			Assert.AreEqual(true, retValue, "Other2");

			#endregion






			#endregion

		}

		// CR2899: This ElgibleForEmergencyIssue method changed in order to make processing more
		// efficient.  Instead of it processing the specimen, the calling method must now to it
		// and pass in booleans that reflect the results.
		//
		// As a result of this change, the corresponding unit tests also changed.
		[Test]
		public void ElgibleForEmergencyIssue_Fail()
		{
			BOL.Specimen tmpS = new BOL.Specimen();

			this._tstCodabar.BloodComponentClassID = (int)COMMON.ComponentClass.WB;
			
			bool retValue = this._tstCodabar.EligibleForEmergencyIssue(false, false, false);
			
			Assert.AreEqual(false, retValue);
		}

		[Test]
		public void OptionalCrossmatch_Pass()
		{
			this._tstISBT.BloodComponentClassId = (int) COMMON.ComponentClass.Other;
			
			this._tstISBT.ProductType.ProductTypeCode = "E034";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E034");
			
			this._tstISBT.ProductType.ProductTypeCode = "E035";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E035");
			
			this._tstISBT.ProductType.ProductTypeCode = "E036";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E036");
			
			this._tstISBT.ProductType.ProductTypeCode = "E037";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E037");
			
			this._tstISBT.ProductType.ProductTypeCode = "E038";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E038");
			
			this._tstISBT.ProductType.ProductTypeCode = "E039";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E039");
			
			this._tstISBT.ProductType.ProductTypeCode = "E042";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E042");
			
			this._tstISBT.ProductType.ProductTypeCode = "E043";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E043");
			
			this._tstISBT.ProductType.ProductTypeCode = "E044";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E044");
			
			this._tstISBT.ProductType.ProductTypeCode = "E045";
			Assert.AreEqual(true, this._tstISBT.OptionalCrossmatch, "E045");
		}

		[Test]
		public void OptionalCrossmatch_Fail()
		{
			this._tstISBT.BloodComponentClassId = (int) COMMON.ComponentClass.Other;
			
			this._tstISBT.ProductType.ProductTypeCode = "E014";
			Assert.AreEqual(false, this._tstISBT.OptionalCrossmatch, "E014");
			
			this._tstISBT.ProductType.ProductTypeCode = "E015";
			Assert.AreEqual(false, this._tstISBT.OptionalCrossmatch, "E015");
			
			this._tstISBT.ProductType.ProductTypeCode = "E017";
			Assert.AreEqual(false, this._tstISBT.OptionalCrossmatch, "E017");
			
			this._tstISBT.ProductType.ProductTypeCode = "E018";
			Assert.AreEqual(false, this._tstISBT.OptionalCrossmatch, "E018");
			
			this._tstISBT.ProductType.ProductTypeCode = "E019";
			Assert.AreEqual(false, this._tstISBT.OptionalCrossmatch, "E019");
			
			this._tstISBT.ProductType.ProductTypeCode = "E040";
			Assert.AreEqual(false, this._tstISBT.OptionalCrossmatch, "E040");
			
			this._tstISBT.ProductType.ProductTypeCode = "E041";
			Assert.AreEqual(false, this._tstISBT.OptionalCrossmatch, "E041");
		}

		[Test]
		public void FullProductCode_Pass()
		{
			//Test Codabar
			string tstValue = this._tstCodabar.FullProductCode;
			Assert.AreEqual(this._tstCodabar.FullProductCode, tstValue, "Codabar-FullProductCode");

			//Test ISBT
			tstValue = this._tstISBT.FullProductCode;
			Assert.AreEqual(this._tstISBT.FullProductCode, tstValue, "ISBT-FullProductCode");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void FullProductCode_Fail()
		{
		}
	
		[Test]
		public void BloodUnitGuid_Pass()
		{
			Guid tstValue = Guid.NewGuid();

			this._tstISBT.BloodUnitGuid = tstValue;

			Assert.AreEqual(tstValue, this._tstISBT.BloodUnitGuid, "BloodUnitGuid");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BloodUnitGuid_Fail()
		{
		}


		[Test]
		public void CheckValidUnitIDForLocalSupplier_Pass()
		{
			//Test Codabar
			string tstValue = string.Empty;
			this._tstCodabar.CheckValidUnitIDForLocalSupplier(this._tstCodabar.EyeReadableUnitId, out tstValue);

			Assert.AreEqual(true, tstValue.Length >= 0, "Codabar-Length");

			//Test ISBT
			tstValue = string.Empty;
			this._tstISBT.CheckValidUnitIDForLocalSupplier(this._tstISBT.EyeReadableUnitId, out tstValue);

			Assert.AreEqual(true, tstValue.Length >= 0, "ISBT-Length");

		}

		[Test]
		public void CheckValidUnitIDForLocalSupplier_Fail()
		{
			//Test Codabar
			string tstValue = string.Empty;
			this._tstCodabar.CheckValidUnitIDForLocalSupplier("2H@@TERS!", out tstValue);

			Assert.AreEqual(true, tstValue.Length > 0, "Codabar-Length");

			//Test ISBT
			tstValue = string.Empty;
			this._tstISBT.CheckValidUnitIDForLocalSupplier("2H@@TERS!", out tstValue);

			Assert.AreEqual(true, tstValue.Length > 0, "ISBT-Length");

		}

		[Test]
		public void LocalSupplier_Pass()
		{
			Guid unitGuid = (Guid)this._dtLocalSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid];
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(unitGuid, false);

			this._tstCodabar.LocalSupplier = tmpLS;

			Assert.AreEqual(tmpLS, this._tstCodabar.LocalSupplier, "LocalSupplier");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LocalSupplier_Fail()
		{
		}


		[Test]
		public void OriginalVolume_Pass()
		{
			decimal tstValue = (decimal)new Random().Next(1,99999);

			this._tstCodabar.OriginalVolume = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.OriginalVolume, "OriginalVolume");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void OriginalVolume_Fail()
		{
		}

		[Test]
		public void DiscardedPlasmaVolume_Pass()
		{
			decimal tstValue = (decimal)new Random().Next(1,99999);

			this._tstCodabar.DiscardedPlasmaVolume = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.DiscardedPlasmaVolume, "DiscardedPlasmaVolume");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void DiscardedPlasmaVolume_Fail()
		{
		}

		[Test]
		public void PooledUnitsCount_Pass()
		{
			short tstValue = this._tstCodabar.PooledUnitsCount;

			Assert.AreEqual(tstValue, this._tstCodabar.PooledUnitsCount, "PooledUnitsCount");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PooledUnitsCount_Fail()
		{
		}

		[Test]
		public void UnitInDate_Pass()
		{
			DateTime tstValue = DateTime.Now;

			this._tstCodabar.UnitInDate = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.UnitInDate);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void UnitInDate_Fail()
		{
		}

		[Test]
		public void UnitContainerLotNumber_Pass()
		{
			string tstValue = new Random().Next(1,9999999).ToString();

			this._tstCodabar.UnitContainerLotNumber = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.UnitContainerLotNumber, "UnitContainerLotNumber");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void UnitContainerLotNumber_Fail()
		{
			this._tstCodabar.UnitContainerLotNumber = "".PadRight(16,'X');
		}

		[Test]
		public void LabelTypeCode_Pass()
		{
			string tstValue = new Random().Next(1,9).ToString();

			this._tstCodabar.LabelTypeCode = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.LabelTypeCode, "LabelTypeCode");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void LabelTypeCode_Fail()
		{
			this._tstCodabar.LabelTypeCode = "".PadRight(2,'X');
		}

		[Test]
		public void LabelUnitId_Pass()
		{
			string tstValue = this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.LabelUnitId].ToString();

			Assert.AreEqual(tstValue, this._tstCodabar.LabelUnitId, "LabelUnitId");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LabelUnitId_Fail()
		{
		}

		[Test]
		public void EyeReadableUnitId_Pass()
		{
			string tstValue = this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.EyeReadableUnitId].ToString();

			Assert.AreEqual(tstValue, this._tstCodabar.EyeReadableUnitId, "EyeReadableUnitId");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void EyeReadableUnitId_Fail()
		{
		}

		[Test]
		public void ProcessingTechId_Pass()
		{
			string tstValue = DateTime.Now.ToString();

			this._tstCodabar.ProcessingTechId = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.ProcessingTechId, "ProcessingTechId");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ProcessingTechId_Fail()
		{
		}

		[Test]
		public void UnitFdaRegistrationNumber_Pass()
		{
			//Test Codabar
			string tstValue = new Random().Next(1111111,9999999).ToString();

			this._tstCodabar.UnitFdaRegistrationNumber = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.UnitFdaRegistrationNumber, "UnitFdaRegistrationNumber");
			Assert.AreEqual(false, this._tstCodabar.IsRuleBroken(BROKEN.UnitFdaRegistrationNumber), "BrokenRule");

			//Test ISBT
			tstValue = new Random().Next(1111111,9999999).ToString();

			this._tstISBT.UnitFdaRegistrationNumber = tstValue;

			Assert.AreEqual(tstValue, this._tstISBT.UnitFdaRegistrationNumber, "UnitFdaRegistrationNumber");
			Assert.AreEqual(false, this._tstISBT.IsRuleBroken(BROKEN.UnitFdaRegistrationNumber), "BrokenRule");

		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void UnitFdaRegistrationNumber_Fail()
		{
			this._tstCodabar.UnitFdaRegistrationNumber = "ABCDEFG";
		}

		
		[Test]
		public void BarcodeType_Pass()
		{
			Assert.AreEqual(COMMON.BarcodeType.Codabar, this._tstCodabar.BarcodeType, "Codabar-BarcodeType");
			Assert.AreEqual(COMMON.BarcodeType.ISBT128, this._tstISBT.BarcodeType, "ISBT-BarcodeType");

			BOL.BloodUnit tstUnit = new BOL.BloodUnit("19740420", "04020", COMMON.BarcodeType.Codabar, this._tstCodabar.BloodProductGuid);

			Assert.AreEqual(COMMON.BarcodeType.Codabar, tstUnit.BarcodeType, "Set BarcodeType");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BarcodeType_Fail()
		{
		}
		
		[Test]
		public void IncomingShipmentGuid_Pass()
		{
			Guid tstValue = Guid.NewGuid();

			this._tstCodabar.IncomingShipmentGuid = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.IncomingShipmentGuid, "IncomingShipmentGuid");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IncomingShipmentGuid_Fail()
		{
		}

		[Test]
		public void EyeReadablePrefix_Pass()
		{
			//EyeReadablePrefix change -- Need a valid LocalSupplier for EyeReadablePrefix to work
			this._tstCodabar.LocalSupplier = new BOL.LocalSupplier(Guid.NewGuid(), true);
			string tstValue = new Random().Next(11,99).ToString();

			this._tstCodabar.EyeReadablePrefix = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.EyeReadablePrefix, "EyeReadablePrefix");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void EyeReadablePrefix_Fail()
		{
			this._tstCodabar.EyeReadablePrefix = new Random().Next(100,999).ToString();
		}
		
		[Test]
		public void EntryMethodCode_Pass()
		{
			string tstValue = new Random().Next(1,9).ToString();

			this._tstCodabar.EntryMethodCode = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.EntryMethodCode, "EntryMethodCode");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void EntryMethodCode_Fail()
		{
			this._tstCodabar.EntryMethodCode = "".PadRight(2,'X');
		}

		[Test]
		public void DirectedPatient_Pass()
		{
			this._tstCodabar.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.Codabar_BloodUnit);

			BOL.Patient tmpP = new BOL.Patient((Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid]);

			this._tstCodabar.DirectedPatient = tmpP;

			Assert.AreEqual(tmpP.PatientGuid, this._tstCodabar.DirectedPatient.PatientGuid, "EntryMethodCode");
			Assert.AreEqual(false, this._tstCodabar.IsRuleBroken(BROKEN.DirectedPatient), "BrokenRule");
		}

		[Test]
		public void DirectedPatient_Fail()
		{
			this._tstCodabar.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.Codabar_BloodUnit);

			this._tstCodabar.DirectedPatient = null;
			Assert.AreEqual(true, this._tstCodabar.IsRuleBroken(BROKEN.DirectedPatient), "BrokenRule");
		}

		[Test]
		public void PatientAssignments_Pass()
		{
			DataTable tstValue = this._tstCodabar.PatientAssignments;
			Assert.AreEqual(true, tstValue.Rows.Count >= 0, "Rows.Count");
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void PatientAssignments_Fail()
		{
			this._tstCodabar.BloodUnitGuid = Guid.Empty;
			DataTable tstValue = this._tstCodabar.PatientAssignments;
		}

		[Test]
		public void CmvNegative_Pass()
		{
			bool tstValue = this._tstCodabar.CmvNegative;

			Assert.AreEqual(tstValue, this._tstCodabar.CmvNegative, "CmvNegative");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void CmvNegative_Fail()
		{
		}

		[Test]
		public void DivisionCode_Pass()
		{
			string tstValue = new Random().Next(11111,99999).ToString();

			this._tstCodabar.DivisionCode = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.DivisionCode, "DivisionCode");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void DivisionCode_Fail()
		{
			this._tstCodabar.DivisionCode = "".PadRight(6,'X');
		}

		[Test]
		public void SickleCellNegative_Pass()
		{
			bool tstValue = this._tstCodabar.SickleCellNegative;

			Assert.AreEqual(tstValue, this._tstCodabar.SickleCellNegative, "SickleCellNegative");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void SickleCellNegative_Fail()
		{
		}

		[Test]
		public void RbcAntigens_Pass()
		{
			string tstValue = this._tstCodabar.RbcAntigens;

			Assert.AreEqual(tstValue, this._tstCodabar.RbcAntigens, "RbcAntigens");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void RbcAntigens_Fail()
		{
		}
		
		[Test]
		public void RecordStatusCode_Pass()
		{
			COMMON.RecordStatusCode tstValue = COMMON.RecordStatusCode.Historical;

			this._tstCodabar.RecordStatusCode = tstValue;
			
			Assert.AreEqual(tstValue, this._tstCodabar.RecordStatusCode, "RecordStatusCode");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RecordStatusCode_Fail()
		{
		}
		
		[Test]
		public void BloodUnitStatus_Pass()
		{
			BOL.BloodUnitStatus tmpBUS = new BOL.BloodUnitStatus(this._tstCodabar.BloodUnitGuid);

			this._tstCodabar.BloodUnitStatus = tmpBUS;

			Assert.AreEqual(tmpBUS, this._tstCodabar.BloodUnitStatus, "BloodUnitStatus");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BloodUnitStatus_Fail()
		{
		}
				
		[Test]
		public void BloodUnitFinance_Pass()
		{
			BOL.BloodUnitFinance tmpBUF = new BOL.BloodUnitFinance(this._tstCodabar.BloodUnitGuid);

			this._tstCodabar.BloodUnitFinance = tmpBUF;

			Assert.AreEqual(tmpBUF, this._tstCodabar.BloodUnitFinance, "BloodUnitFinance");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BloodUnitFinance_Fail()
		{
		}
				
		[Test]
		public void BloodUnitMedia_Pass()
		{
			BOL.BloodUnitMedia tmpBUM = new BOL.BloodUnitMedia(this._tstCodabar.BloodUnitGuid);

			this._tstCodabar.BloodUnitMedia = tmpBUM;

			Assert.AreEqual(tmpBUM, this._tstCodabar.BloodUnitMedia, "BloodUnitMedia");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BloodUnitMedia_Fail()
		{
		}
				
		[Test]
		public void PatientSpecimens_Pass()
		{
			BOL.Specimen[] tstValue = 
				{
					new BOL.Specimen(),
					new BOL.Specimen()
				};

			this._tstCodabar.PatientSpecimens = tstValue;

			Assert.AreEqual(tstValue, this._tstCodabar.PatientSpecimens, "PatientSpecimens");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PatientSpecimens_Fail()
		{
		}
				
		[Test]
		public void AssignedToPatients_Pass()
		{
			System.Collections.ArrayList tstValue = this._tstCodabar.AssignedToPatients;

			Assert.IsNotNull(tstValue, "Object");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void AssignedToPatients_Fail()
		{
		}
				
		[Test]
		public void CheckCharacter_Pass()
		{
			this._tstISBT.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.ISBT128_BloodUnit);

			string tstValue = new Random().Next(1,9).ToString();

			this._tstISBT.CheckCharacter = tstValue;

			Assert.AreEqual(tstValue, this._tstISBT.CheckCharacter, "CheckCharacter");
			Assert.AreEqual(false, this._tstISBT.IsRuleBroken(BROKEN.CheckCharacter), "BrokenRule");
		}

		[Test]
		public void CheckCharacter_Fail()
		{
			this._tstISBT.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.ISBT128_BloodUnit);
			this._tstISBT.CheckCharacter = string.Empty;

			Assert.AreEqual(true, this._tstISBT.IsRuleBroken(BROKEN.CheckCharacter), "BrokenRule");
		}

		[Test]
		public void IsEligibleForCrossover_Pass()
		{
			string tstValue = this._tstCodabar.IsEligibleForCrossover();

			Assert.AreEqual(true, tstValue.Length >= 0, "Length");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsEligibleForCrossover_Fail()
		{
		}
						
		[Test]
		public void IsAutologous_Pass()
		{
			char tstValue;

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.AutologousEligibleForCrossover);
			Assert.AreEqual(true, BOL.BloodUnit.IsAutologous(tstValue), "AutologousEligibleForCrossover");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.ForAutologousUseOnly);
			Assert.AreEqual(true, BOL.BloodUnit.IsAutologous(tstValue), "ForAutologousUseOnly");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.ForAutologousUseOnlyBiohazardous);
			Assert.AreEqual(true, BOL.BloodUnit.IsAutologous(tstValue), "ForAutologousUseOnlyBiohazardous");
		}


		[Test]
		public void IsAutologous_Fail()
		{
			char tstValue;

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.PaidAllogeneicCollection);
			Assert.AreEqual(false, BOL.BloodUnit.IsAutologous(tstValue), "PaidAllogeneicCollection");
		}
						
		[Test]
		public void IsAllogenic_Pass()
		{
			char tstValue;

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.VolunteerAllogeneicDonor);
			Assert.AreEqual(true, BOL.BloodUnit.IsAllogenic(tstValue), "VolunteerAllogeneicDonor");
		}

		[Test]
		public void IsAllogenic_Fail()
		{
			char tstValue;

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.PaidAllogeneicCollection);
			Assert.AreEqual(false, BOL.BloodUnit.IsAllogenic(tstValue), "PaidAllogeneicCollection");
		}
						
		[Test]
		public void IsDirected_Pass()
		{
			char tstValue;

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.VolunteerDirectedEligibeForCrossover);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "VolunteerDirectedEligibeForCrossover");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.PaidDirectedEligibleForCrossover);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "PaidDirectedEligibleForCrossover");
			
			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.ForDirectedDonorUseOnly);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "ForDirectedDonorUseOnly");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.ForDirectedDonorUseOnlyLimitedExposure);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "ForDirectedDonorUseOnlyLimitedExposure");
			
			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.ForDirectedDonorUseOnlyMedicalException);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "ForDirectedDonorUseOnlyMedicalException");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.ForDirectedDonorUseOnlyBiohazardous);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "ForDirectedDonorUseOnlyBiohazardous");
			
			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.DedicatedDonor);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "DedicatedDonor");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.DesignatedDonor);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "DesignatedDonor");
			
			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.VolunteerResearchDonor);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "VolunteerResearchDonor");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.VolunteerSourceDonor);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "VolunteerSourceDonor");
			
			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.VolunteerTherapeuticCollection);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "VolunteerTherapeuticCollection");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.PaidAllogeneicCollection);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "PaidAllogeneicCollection");
			
			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.PaidResearchCollection);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "PaidResearchCollection");

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.PaidSourceCollection);
			Assert.AreEqual(true, BOL.BloodUnit.IsDirected(tstValue), "PaidSourceCollection");
			
		}

		[Test]
		public void IsDirected_Fail()
		{
			char tstValue;

			tstValue = COMMON.Utility.GetDonationTypeCodeCharFromEnum(COMMON.DonationType.AutologousEligibleForCrossover);
			Assert.AreEqual(false, BOL.BloodUnit.IsDirected(tstValue), "AutologousEligibleForCrossover");
		}

		[Test]
		public void GetBloodUnitGuid_Pass()
		{
			Guid unitGuid = BOL.BloodUnit.GetBloodUnitGuid(this._tstCodabar.EyeReadableUnitId, this._tstCodabar.ProductCode, UnitTestConfig.TestDivisionCode, this._tstCodabar.UnitDivisionId);

			Assert.AreEqual(this._tstCodabar.BloodUnitGuid, unitGuid, "BloodUnitGuid");
		}

		[Test]
		[ExpectedException(typeof(Common.DataAccessLayerException))]
		public void GetBloodUnitGuid_Fail()
		{
			BOL.BloodUnit.GetBloodUnitGuid("ABC", "ABC", this.TestDivisionCode, "ABC");
		}

		[Test]
		public void GetBloodUnitsByIdAndProduct_Pass()
		{
			//Test Codabar
			DataTable dtCodabar = BOL.BloodUnit.GetBloodUnitsByIdAndProduct
			(
				this._tstCodabar.EyeReadableUnitId,
				this._tstCodabar.ProductCode, 
				this.TestDivisionCode,
				this._tstCodabar.UnitDivisionId,
				this._tstCodabar.DonationType.DonationTypeCode.ToString(),
				COMMON.AdditionalBloodUnitSearchCriteria.InactiveUnits 
			);

			Assert.AreEqual(true, dtCodabar.Rows.Count > 0, "Codabar-Rows.Count");
			Assert.AreEqual(this._tstCodabar.BloodUnitGuid, (Guid)dtCodabar.Rows[0][TABLES.BloodUnit.BloodUnitGuid], "Codabar-BloodUnitGuid");

			//Test ISBT
			DataTable dtISBT = BOL.BloodUnit.GetBloodUnitsByIdAndProduct
				(
				this._tstISBT.EyeReadableUnitId,
				this._tstISBT.ProductCode, 
				this.TestDivisionCode,
				this._tstISBT.UnitDivisionId,
				this._tstISBT.DonationType.DonationTypeCode.ToString(),
				COMMON.AdditionalBloodUnitSearchCriteria.InactiveUnits 
				);

			Assert.AreEqual(true, dtISBT.Rows.Count > 0, "ISBT-Rows.Count");
			Assert.AreEqual(this._tstISBT.BloodUnitGuid, (Guid)dtISBT.Rows[0][TABLES.BloodUnit.BloodUnitGuid], "ISBT-BloodUnitGuid");

		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetBloodUnitsByIdAndProduct_Fail()
		{
			DataTable units = BOL.BloodUnit.GetBloodUnitsByIdAndProduct
			(
				string.Empty, string.Empty, this.TestDivisionCode, string.Empty, string.Empty, COMMON.AdditionalBloodUnitSearchCriteria.None
			);
		}

		[Test]
		public void GetBloodUnitDetails_Pass()
		{
			//Test Codabar
			DataTable dtCodabar = BOL.BloodUnit.GetBloodUnitDetails(this._tstCodabar.EyeReadableUnitId, this._tstCodabar.BloodProductGuid);
			Assert.AreEqual(true, dtCodabar.Rows.Count > 0, "Codabar-Rows.Count");

			//Test ISBT
			DataTable dtISBT = BOL.BloodUnit.GetBloodUnitDetails(this._tstISBT.EyeReadableUnitId, this._tstISBT.BloodProductGuid);
			Assert.AreEqual(true, dtISBT.Rows.Count > 0, "ISBT-Rows.Count");

		}

		[Test]
		public void GetBloodUnitDetails_Fail()
		{
			DataTable dtTest = BOL.BloodUnit.GetBloodUnitDetails("ABC", Guid.Empty);
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}
		[Test]
		public void BloodUnitExists_Overload_Guid_Pass()
		{
			//TEST Codabar
			Guid codabarGuid = Guid.Empty;

			bool tstValue = BOL.BloodUnit.BloodUnitExists(this._tstCodabar.EyeReadableUnitId, this._tstCodabar.BloodProductGuid, this._tstCodabar.ProductCode, ref codabarGuid);

			Assert.AreEqual(true, tstValue, "Codabar-ReturnValue");
			Assert.AreEqual(codabarGuid, this._tstCodabar.BloodUnitGuid, "Codabar-BloodUnitGuid");

			//TEST ISBT
			Guid isbtGuid = Guid.Empty;

			bool tstValue2 = BOL.BloodUnit.BloodUnitExists(this._tstISBT.EyeReadableUnitId, this._tstISBT.BloodProductGuid, this._tstISBT.ProductCode, ref isbtGuid);

			Assert.AreEqual(true, tstValue2, "ISBT-ReturnValue");
			Assert.AreEqual(isbtGuid, this._tstISBT.BloodUnitGuid, "ISBT-BloodUnitGuid");
		}
		[Test]
		public void BloodUnitExists_Overload_Guid_Fail()
		{
			Guid unitGuid = Guid.Empty;

			bool tstValue = BOL.BloodUnit.BloodUnitExists("Simpson", Guid.Empty , "Simpson", ref unitGuid);
			Assert.AreEqual(false, tstValue, "ReturnValue");
		}

		[Test]
		public void IncomingShipmentBloodUnitExists_Pass()
		{
			//Test Codabar
			Guid codabarGuid = Guid.Empty;

			bool tstValue = BOL.BloodUnit.IncomingShipmentBloodUnitExists(this._tstCodabar.EyeReadableUnitId, this._tstCodabar.BloodProductGuid, this._tstCodabar.FullProductCode, ref codabarGuid);

			Assert.AreEqual(true, tstValue, "Codabar-ReturnValue");
			Assert.AreEqual(codabarGuid, this._tstCodabar.BloodUnitGuid, "Codabar-BloodUnitGuid");

			//Test ISBT
			Guid isbtGuid = Guid.Empty;

			bool tstValue2 = BOL.BloodUnit.IncomingShipmentBloodUnitExists(this._tstISBT.EyeReadableUnitId, this._tstISBT.BloodProductGuid, this._tstCodabar.FullProductCode, ref isbtGuid);

			Assert.AreEqual(true, tstValue2, "ISBT-ReturnValue");
			Assert.AreEqual(isbtGuid, this._tstISBT.BloodUnitGuid, "ISBT-BloodUnitGuid");
		}

		[Test]
		public void IncomingShipmentBloodUnitExists_Fail()
		{
			Guid unitGuid = Guid.Empty;

			bool tstValue = BOL.BloodUnit.IncomingShipmentBloodUnitExists("Simpson", Guid.Empty , "Simpson", ref unitGuid);
			Assert.AreEqual(false, tstValue, "ReturnValue");
		}


		[Test]
		public void BloodUnitExists_String_Guid_Fail()
		{
			Guid unitGuid = Guid.Empty;

			bool tstValue = BOL.BloodUnit.BloodUnitExists(unitGuid);
			Assert.AreEqual(false, tstValue, "ReturnValue");
		}

		[Test]
		public void GetUnitDivisionId_Pass()
		{
			string tstValue = new Random().Next(11,99).ToString();
			string productCode = "123456" + tstValue;

			Assert.AreEqual(tstValue, BOL.BloodUnit.GetUnitDivisionId(productCode), "UnitDivisionId");
		}

		[Test]
		public void GetUnitDivisionId_Fail()
		{
			string productCode = "123456";

			Assert.AreEqual(string.Empty, BOL.BloodUnit.GetUnitDivisionId(productCode), "UnitDivisionId");
		}

		[Test]
		public void BloodUnitExists_Guid_Pass()
		{
			//Test Codabar
			Assert.AreEqual(true, BOL.BloodUnit.BloodUnitExists(this._codabarGuid), "Codabar-ReturnValue");

			//Test ISBT
			Assert.AreEqual(true, BOL.BloodUnit.BloodUnitExists(this._isbtGuid), "ISBT-ReturnValue");
		}

		[Test]
		public void BloodUnitExists_Guid_Fail()
		{
			Assert.AreEqual(false, BOL.BloodUnit.BloodUnitExists(Guid.NewGuid()), "ReturnValue");
		}

		[Test]
		public void GetLimitedBloodUnit_Pass()
		{
			Guid shipmentGuid = Guid.Empty;

			DataTable dtShipment = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 IncomingShipmentGuid FROM vw_BloodShipAllCurrent WHERE UnitStatusCode = 'L' AND DivisionCode = '" + this.TestDivisionCode + "'", false);
			if (dtShipment.Rows.Count == 0)
			{
				DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, DataCreator.GetRandomBloodProduct(true, COMMON.ComponentClass.RBC).BloodProductGuid, true);
				shipmentGuid = DataCreator.GetRecordGuid("SELECT TOP 1 IncomingShipmentGuid FROM vw_BloodShipAllCurrent WHERE UnitStatusCode = 'L' AND DivisionCode = '" + this.TestDivisionCode + "'");
			}

			DataTable dtTest = BOL.BloodUnit.GetLimitedBloodUnit(shipmentGuid, true, true);

			Assert.AreEqual(true, dtTest.Rows.Count > 0, "No Limited BloodUnit");
		}

		[Test]
		public void GetLimitedBloodUnit_Fail()
		{
			Guid shipmentGuid = Guid.NewGuid();

			DataTable dtTest = BOL.BloodUnit.GetLimitedBloodUnit(shipmentGuid, true, true);

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetLimitedBloodUnitsByDivision_Pass()
		{
			DataTable dtTest = BOL.BloodUnit.GetLimitedBloodUnitsByDivision(this.TestDivisionCode, false, true);

			if (dtTest.Rows.Count == 0)
			{
				DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, DataCreator.GetRandomBloodProduct(true, COMMON.ComponentClass.RBC).BloodProductGuid, true);
				dtTest = BOL.BloodUnit.GetLimitedBloodUnitsByDivision(this.TestDivisionCode, false, true);
			}

			Assert.AreEqual(false, (dtTest.Rows.Count == 0), "No Limited BloodUnits");
		}

		[Test]
		public void GetLimitedBloodUnitsByDivision_Fail()
		{
			DataTable dtTest = BOL.BloodUnit.GetLimitedBloodUnitsByDivision("INDY", false, true);

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetPhenotypeUnits_Pass()
		{
			DataTable dtTest = BOL.BloodUnit.GetPhenotypeUnits(this.TestDivisionCode, false);

			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
		}

		[Test]
		public void GetPhenotypeUnits_Fail()
		{
			DataTable dtTest = BOL.BloodUnit.GetPhenotypeUnits("INDY", false);

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetBloodUnits_Pass()
		{
			DataTable dtTest = BOL.BloodUnit.GetBloodUnits(this._tstCodabar.IncomingShipmentGuid);

			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
		}

		[Test]
		public void GetBloodUnits_Fail()
		{
			DataTable dtTest = BOL.BloodUnit.GetBloodUnits(Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}


		[Test]
		public void GetBloodUnitsForEdit_Pass()
		{
			DataTable dtTest = BOL.BloodUnit.GetBloodUnitsForEdit(true);
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows1.Count");

			DataTable dtTest2 = BOL.BloodUnit.GetBloodUnitsForEdit(false);
			Assert.AreEqual(true, dtTest2.Rows.Count >= 0, "Rows2.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetBloodUnitsForEdit_Fail()
		{
		}
		
		[Test]
		public void GetBloodTypeRHFactorFromBloodTypeCode_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			Guid patientGuid = (Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			Guid bloodProductGuid = (Guid)this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.BloodProductGuid];

			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), patientGuid, bloodProductGuid, true);
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);

			DataRow drTest = BOL.BloodUnit.GetBloodTypeRHFactorFromBloodTypeCode("06");
			Assert.IsNotNull(drTest, "Object");
		}

		[Test]
		[ExpectedException(typeof(IndexOutOfRangeException))]
		public void GetBloodTypeRHFactorFromBloodTypeCode_Fail()
		{
			BOL.BloodUnit.GetBloodTypeRHFactorFromBloodTypeCode("Q");
		}

		[Test]
		public void GetEmptyBloodUnitDataTable_Pass()
		{
			DataTable dtTest = BOL.BloodUnit.GetEmptyBloodUnitDataTable();
			Assert.AreEqual(40, dtTest.Columns.Count, "Columns.Count");
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetEmptyBloodUnitDataTable_Fail()
		{
		}

		[Test]
		public void GetBloodUnitsInFinalStatus_Pass()
		{
			DataTable dtTest = BOL.BloodUnit.GetBloodUnitsInFinalStatus(true);
			if (dtTest.Rows.Count == 0)
			{
				//"M" Status units are Splits and Pools
				BOL.BloodUnit tmpNewUnit = DataCreator.CreateBloodUnit(true);

				BOL.BloodUnitModification tmpBUM = new BOL.BloodUnitModification(Common.ProductModification.SplitAdult, Common.ModifyUnitMethod.Closed, DateTime.Now.AddDays(-1), Common.LogonUser.LogonUserName);
				tmpBUM.SplitUnit(tmpNewUnit.BloodUnitGuid, 2);
				tmpBUM.Save(Common.UpdateFunction.UnitTests);

				dtTest = BOL.BloodUnit.GetBloodUnitsInFinalStatus(true);
			}
			//Modified Units are included
			Assert.AreEqual(false, dtTest.Rows.Count == 0, "No BloodUnits in Final Status #1");

			//Modified units are filtered out
			DataTable dtTest2 = BOL.BloodUnit.GetBloodUnitsInFinalStatus(false);
			if (dtTest2.Rows.Count == 0)
			{
				//The simplest way is to discard a unit
				Guid discardGuid = DataCreator.GetRandomRecordGuid(TABLES.CannedComment.TableName, "");
				Guid unitGuid = Guid.NewGuid();
				DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, DataCreator.GetRandomBloodProduct(true, Common.ComponentClass.ConversionClass).BloodProductGuid, true);

				UnitTestUtil.RunSQL("UPDATE BloodUnitStatus SET DiscardCannedCommentGuid = '" + discardGuid.ToString() + "' WHERE BloodUnitGuid = '" + unitGuid.ToString() + "'");

				dtTest2 = BOL.BloodUnit.GetBloodUnitsInFinalStatus(false);
			}
			Assert.AreEqual(false, dtTest2.Rows.Count == 0, "No BloodUnits in Final Status #2");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetBloodUnitsInFinalStatus_Fail()
		{
		}

        [Test]
        public void IsComponentRequirementSatisfied_Pass()
        {
            // CR3206
            // In this testing, all requirements are met. While the units are manually manipulated below,
            // all data represents units that actually could exist in the system.
            //
            //
            // WBs
            //      #1: IRRADIATED       Check for C2 attribute 
            //      #2: LEUKOREDUCED     Check for D* attribute except D1
            //      #3: CMV              Check for negative test result
            //      #4: Sickle Cell      Check for negative test result
            //
            //
            // RBCs
            //      #1: IRRADIATED       Check for C2 attribute 
            //      #2: LEUKOREDUCED     Check for D* attribute except D1
            //      #3: WASHED RBC       Check for conforming Product Type
            //      #4: CMV              Check for negative test result
            //      #5: Sickle Cell      Check for negative test result
            //
            //
            // PLATELETS
            //      #1: IRRADIATED       Check for C2 attribute 
            //      #2: LEUKOREDUCED     Check for D* attribute except D1
            //      #3: WASHED PLT       Check for conforming Product Type
            //      #4: CMV              Check for negative test result
            //
            //
            // OTHER
            //      #1: IRRADIATED       Check for C2 attribute 
            //      #2: CMV              Check for negative test result
            //
            //

            ////////////////////////////////////////////////////////
            ///			WBs		WBs		WBs		WBs		WBs	
            ////////////////////////////////////////////////////////

            // #1 Irradiated, check for C2 attribute
            this._tstISBT.ProductType.ProductTypeCode = "E001";
            this._tstISBT.ProductCode = "E0106";
            this._tstISBT.ProductAttributeForm = "@11 C2";
            bool tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(tstValue);



            // #2a Leukoreduced, check for D* attributes
            this._tstISBT.ProductType.ProductTypeCode = "E001";
            this._tstISBT.ProductCode = "E0046";
            this._tstISBT.ProductAttributeForm = "@05 C2 D4";
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(tstValue);



            // #2b Leukoreduced, check for D* attributes
            this._tstISBT.ProductType.ProductTypeCode = "E001";
            this._tstISBT.ProductCode = "E5438";
            this._tstISBT.ProductAttributeForm = "@BV C2 DC";
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(tstValue);



            // Setup for CMV and SC
            Guid bloodUnitGuid1 = Guid.NewGuid();
            DataRow drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E0106'");
            BOL.BloodUnit unit = DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, (Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnitGuid1, 103, "N");	// CMV Negative
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnitGuid1, 104, "N");	// Sickle cell



            // #3 CMV Neg
            tstValue = unit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(tstValue);

            // #4 Sickle Cell Neg
            tstValue = unit.IsComponentRequirementSatisfied(Common.ComponentRequirement.SCNEG);
            Assert.IsTrue(tstValue);




            ////////////////////////////////////////////////////////
            ///			RBCs	RBCs	RBCs	RBCs	RBCs	
            ////////////////////////////////////////////////////////

            // #1 Irradiated, check for C2 attribute
            this._tstISBT.ProductType.ProductTypeCode = "E002";
            this._tstISBT.ProductCode = "E0448";
            this._tstISBT.ProductAttributeForm = "@25 C2";
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(tstValue);



            // #2a Leukoreduced, check for D* attributes
            this._tstISBT.ProductType.ProductTypeCode = "E002";
            this._tstISBT.ProductCode = "E4912";
            this._tstISBT.ProductAttributeForm = "@06 C2 D2 H4";
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(tstValue);

            // #2b Leukoreduced, check for D* attributes
            this._tstISBT.ProductType.ProductTypeCode = "E002";
            this._tstISBT.ProductCode = "E5260";
            this._tstISBT.ProductAttributeForm = "@24 C2 DA";
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(tstValue);



            // #3 Washed RBC, check for all 6 product types
            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedRedBloodCells);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
            Assert.IsTrue(tstValue);

            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRedBloodCells);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
            Assert.IsTrue(tstValue);

            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRejuvenatedRedBloodCells);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
            Assert.IsTrue(tstValue);

            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedApheresisRedBloodCells);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
            Assert.IsTrue(tstValue);

            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedApheresisRedBloodCells);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
            Assert.IsTrue(tstValue);

            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRejuvenatedApheresisRedBloodCells);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
            Assert.IsTrue(tstValue);


            bloodUnitGuid1 = Guid.NewGuid();
            drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E0291'");
            unit = DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, (Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnitGuid1, 103, "N");	// CMV Negative
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnitGuid1, 104, "N");	// Sickle cell



            // #4 CMV Neg
            tstValue = unit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(tstValue);

            // #5 Sickle Cell Neg
            tstValue = unit.IsComponentRequirementSatisfied(Common.ComponentRequirement.SCNEG);
            Assert.IsTrue(tstValue);





            ////////////////////////////////////////////////////////
            ///		PLATELETs	PLATELETs	PLATELETs			
            ////////////////////////////////////////////////////////


            // #1 Irradiated, check for C2 attribute
            this._tstISBT.ProductType.ProductTypeCode = "E020";
            this._tstISBT.ProductCode = "E5466";
            this._tstISBT.ProductAttributeForm = "@C4 C2";
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(tstValue);


            // #2 Leukoreduced, check for D* attribute
            this._tstISBT.ProductType.ProductTypeCode = "E020";
            this._tstISBT.ProductCode = "E2864";
            this._tstISBT.ProductAttributeForm = "@69 C2 D6";
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(tstValue);


            // #3 Washed PLATELET, check for all 3 product types
            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedPlatelets);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDPLT);
            Assert.IsTrue(tstValue);

            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedPooledPlatelets);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDPLT);
            Assert.IsTrue(tstValue);

            this._tstISBT.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedApheresisPlatelets);
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDPLT);
            Assert.IsTrue(tstValue);


            // #4 CMV Neg setup
            bloodUnitGuid1 = Guid.NewGuid();
            drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E5464'");
            unit = DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, (Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnitGuid1, 103, "N");	// CMV Negative


            // #5 CMV Neg
            tstValue = unit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(tstValue);



            ////////////////////////////////////////////////////////
            ///		OTHER		OTHER		OTHER		
            ////////////////////////////////////////////////////////
            ///
            // OTHER
            //      #10: IRRADIATED       Check for C2 attribute 
            //      #11: CMV              Check for negative test result



            // #1 Irradiated, check for C2 attribute
            this._tstISBT.ProductType.ProductTypeCode = "E034";
            this._tstISBT.ProductCode = "E6113";
            this._tstISBT.ProductAttributeForm = "@BT C2";
            tstValue = this._tstISBT.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(tstValue);


            // #2 CMV Neg setup
            bloodUnitGuid1 = Guid.NewGuid();
            drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E3805'");
            unit = DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, (Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnitGuid1, 103, "N");	// CMV Negative


            // #2 CMV Neg
            tstValue = unit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(tstValue);
        }


        [Test]
        public void IsComponentRequirementSatisfied_Fail()
        {
            // CR3206
            // In this testing, requirements are not met. While the units are manually manipulated below,
            // all data represents units that actually could exist in the system.
            //
            //
            // WBs
            //      #1: IRRADIATED       Check for C2 attribute 
            //      #2: LEUKOREDUCED     Check for D* attribute except D1
            //		#3: WASHED RBC		 No WB units qualify 
            //      #4: CMV              Check for negative test result
            //      #5: Sickle Cell      Check for negative test result
            //
            //
            // RBCs
            //      #1: IRRADIATED       Check for C2 attribute 
            //      #2: LEUKOREDUCED     Check for D* attribute except D1
            //      #3: WASHED RBC       Check for conforming Product Type
            //      #4: CMV              Check for negative test result
            //      #5: Sickle Cell      Check for negative test result
            //
            //
            // PLATELETS
            //      #1: IRRADIATED       Check for C2 attribute 
            //      #2: LEUKOREDUCED     Check for D* attribute except D1
            //      #3: WASHED PLT       Check for conforming Product Type
            //      #4: CMV              Check for negative test result
            //
            //
            // OTHER
            //      #1: IRRADIATED       Check for C2 attribute 
            //      #2: CMV              Check for negative test result



            ////////////////////////////////////////////////////////
            ///			WBs		WBs		WBs		WBs		WBs	
            ////////////////////////////////////////////////////////

            System.Guid bpGuid = DataCreator.GetRecordGuid("SELECT BloodProductGuid FROM BloodProduct WHERE ProductCode = 'E6507'");
            BOL.BloodUnit WBUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, bpGuid, false);
            // #1a Irradiated, no C2 attribute
            WBUnit.ProductType.ProductTypeCode = "E001";
            WBUnit.ProductCode = "E6508";
            WBUnit.ProductAttributeForm = "@D1 D5";		// No C2 attribute
            bool tstValue = WBUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(!tstValue, "IrradiateCheck-WB-NoC2");

            // #1b Irradiated, no C2 attribute with false C2
            WBUnit.ProductType.ProductTypeCode = "E001";
            WBUnit.ProductCode = "E6508";
            WBUnit.ProductAttributeForm = "@C2";		// C2, but this isn't an attribute (Note: these units don't actually exist in the system at this time)
            tstValue = WBUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(!tstValue, "IrradiateCheck-WB-WithC2");



            // #2a Leukoreduced, D1 attribute (note that this is a ficticious unit since there are no D1s in the database)
            WBUnit.ProductType.ProductTypeCode = "E001";
            WBUnit.ProductCode = "E6508";
            WBUnit.ProductAttributeForm = "@06 C2 D1 H4";	// D1 is not leukoreduced
            tstValue = WBUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(!tstValue, "LEUKORED-WB-WithD1");

            // #2b Leukoreduced, check for D* attributes with false D*
            WBUnit.ProductType.ProductTypeCode = "E001";
            WBUnit.ProductCode = "E6507";
            WBUnit.ProductAttributeForm = "@DA";// @D* is not leukoreduced
            tstValue = WBUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(!tstValue, "LEUKORED-WB-WithD*");



            // #3 Washed RBC, check product types: The only WB type is E001
            WBUnit.ProductType.ProductTypeCode = "E001";
            WBUnit.ProductCode = "E6507";
            WBUnit.ProductAttributeForm = "@DA";// @D* is not leukoreduced
            tstValue = WBUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
            Assert.IsTrue(!tstValue, "WASHEDRBC-WB-WithD*");



            // #4a CMV when no results
            //Guid bloodUnitGuid1 = Guid.NewGuid();
            //DataRow drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E6507'");
            //BOL.BloodUnit unit = DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, (Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
            tstValue = WBUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(!tstValue, "CMVNeg-WB-NoResults");

            // #4b CMV when a positive result
            DataCreator.CreateBloodUnitTest(Guid.Empty, WBUnit.BloodUnitGuid, 103, "P");	// CMV Negative
            tstValue = WBUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(!tstValue, "CMVNEG-WB-PositiveResult");

            // #5 Sickle Cell Neg when no results
            tstValue = WBUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.SCNEG);
            Assert.IsTrue(!tstValue, "SCNEG-WB-NoResults");

            // #5a Sickle Cell when a positive result
            DataCreator.CreateBloodUnitTest(Guid.Empty, WBUnit.BloodUnitGuid, 104, "P");	// Sickle cell
            tstValue = WBUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.SCNEG);
            Assert.IsTrue(!tstValue, "SCNEG-WB-PositiveResult");



            ////////////////////////////////////////////////////////
            ///			RBCs	RBCs	RBCs	RBCs	RBCs	
            ////////////////////////////////////////////////////////

            // #1a Irradiated, no C2 attribute
            bpGuid = DataCreator.GetRecordGuid("SELECT BloodProductGuid FROM BloodProduct WHERE ProductCode = 'E0291'");
            BOL.BloodUnit RBCUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, bpGuid, false);

            RBCUnit.ProductType.ProductTypeCode = "E002";
            RBCUnit.ProductCode = "E0448";
            RBCUnit.ProductAttributeForm = "@18";		// No C2 attribute
            tstValue = RBCUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(!tstValue, "IRRADIATE-RBC-NoC2");

            // #1b Irradiated, no C2 attribute with false C2
            RBCUnit.ProductType.ProductTypeCode = "E002";
            RBCUnit.ProductCode = "E0448";
            RBCUnit.ProductAttributeForm = "@C2";		// C2, but this isn't an attribute
            tstValue = RBCUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(!tstValue, "IRRADIATE-RBC-WithC2");



            // #2a Leukoreduced, D1 attribute (note that this is a ficticious unit since there are no D1s in the database)
            RBCUnit.ProductType.ProductTypeCode = "E002";
            RBCUnit.ProductCode = "E4912";
            RBCUnit.ProductAttributeForm = "@06 C2 D1 H4";	// D1 is not leukoreduced
            tstValue = RBCUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(!tstValue, "LEUKORED-RBC-WithD1");

            // #2b Leukoreduced, check for D* attributes with false D*
            RBCUnit.ProductType.ProductTypeCode = "E002";
            RBCUnit.ProductCode = "E5260";
            RBCUnit.ProductAttributeForm = "@DA";// @D* is not leukoreduced
            tstValue = RBCUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(!tstValue, "LEUKORED-RBC-WithD*");



            // #3 Washed RBC, check product types
            // Build an arraylist of all RBC prodcut types that aren't washed
            ArrayList nonWashedProducts = new ArrayList();
            nonWashedProducts.Add(Common.ProductType.WholeBlood);							// E001
            nonWashedProducts.Add(Common.ProductType.RedBloodCells);						// E002
            nonWashedProducts.Add(Common.ProductType.FrozenRedBloodCells);					// E004
            nonWashedProducts.Add(Common.ProductType.FrozenRejuvenatedRedBloodCells);		// E005
            nonWashedProducts.Add(Common.ProductType.RejuvenatedRedBloodCells);				// E008
            nonWashedProducts.Add(Common.ProductType.ApheresisRedBloodCells);				// E009

            nonWashedProducts.Add(Common.ProductType.FrozenApheresisRedBloodCells);			// E050
            nonWashedProducts.Add(Common.ProductType.FrozenRedBloodCells);					// E052
            nonWashedProducts.Add(Common.ProductType.FrozenRejuvenatedRedBloodCells);		// E053

            for (int i = 0; i < nonWashedProducts.Count; i++)
            {
                RBCUnit.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum((Common.ProductType)nonWashedProducts[i]);
                tstValue = RBCUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
                Assert.IsTrue(!tstValue, "WASHEDRBC-RBC");
            }



            // #4a CMV when no results
            //bloodUnitGuid1 = Guid.NewGuid();
            //drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E0291'");
            //unit = DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, (Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
            tstValue = RBCUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(!tstValue, "CMVNEG-RBC-NoResults");

            // #4b CMV when a positive result
            DataCreator.CreateBloodUnitTest(Guid.Empty, RBCUnit.BloodUnitGuid, 103, "P");	// CMV Negative
            tstValue = RBCUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(!tstValue, "CMVNEG-RBC=WithResults");



            // #5 Sickle Cell Neg when no results
            tstValue = RBCUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.SCNEG);
            Assert.IsTrue(!tstValue, "SCNEG-RBC-NoResults");

            // #5a Sickle Cell when a positive result
            DataCreator.CreateBloodUnitTest(Guid.Empty, RBCUnit.BloodUnitGuid, 104, "P");	// Sickle cell
            tstValue = RBCUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.SCNEG);
            Assert.IsTrue(!tstValue, "SCNEG-RBC-WithResults");




            ////////////////////////////////////////////////////////
            ///		PLATELETs	PLATELETs	PLATELETs			
            ////////////////////////////////////////////////////////

            bpGuid = DataCreator.GetRecordGuid("SELECT BloodProductGuid FROM BloodProduct WHERE ProductCode = 'E5464'");
            BOL.BloodUnit PLTUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, bpGuid, false);


            // #1a Irradiated, check for C2 attribute
            PLTUnit.ProductType.ProductTypeCode = "E020";
            PLTUnit.ProductCode = "E5464";
            PLTUnit.ProductAttributeForm = "@C4"; 	// No C2 attribute
            tstValue = PLTUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(!tstValue, "IRRADIATE-PLT-NoC2");

            // #1b Irradiated, no C2 attribute with false C2
            PLTUnit.ProductType.ProductTypeCode = "E020";
            PLTUnit.ProductCode = "E5464";
            PLTUnit.ProductAttributeForm = "@C2"; 	// C2, but this isn't an attribute
            tstValue = PLTUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(!tstValue, "IRRADIATE-PLT-WithC2");



            // #2a Leukoreduced, D1 attribute (note that this is a ficticious unit since there are no D1s in the database)
            PLTUnit.ProductType.ProductTypeCode = "E020";
            PLTUnit.ProductCode = "E5464";
            PLTUnit.ProductAttributeForm = "@C4 D1";
            tstValue = PLTUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(!tstValue, "LEUKORED-PLT-WithD1");

            // #2b Leukoreduced, check for D* attributes with false D*
            PLTUnit.ProductType.ProductTypeCode = "E020";
            PLTUnit.ProductCode = "E5464";
            PLTUnit.ProductAttributeForm = "@DA";
            tstValue = PLTUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.LEUKORED);
            Assert.IsTrue(!tstValue, "LEUKORED-PLT-WithD*");



            // #3 Washed PLATELET, check product types
            // Build an arraylist of all RBC prodcut types that aren't washed
            ArrayList nonWashedPlatelets = new ArrayList();
            nonWashedProducts.Add(Common.ProductType.Platelets);							// E020
            nonWashedProducts.Add(Common.ProductType.PooledPlatelets);						// E022
            nonWashedProducts.Add(Common.ProductType.ApheresisPlatelets);					// E024
            nonWashedProducts.Add(Common.ProductType.FrozenApheresisPlatelets);				// E025
            nonWashedProducts.Add(Common.ProductType.ThawedApheresisPlatelets);				// E026

            for (int i = 0; i < nonWashedPlatelets.Count; i++)
            {
                PLTUnit.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum((Common.ProductType)nonWashedProducts[i]);
                tstValue = PLTUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.WASHEDRBC);
                Assert.IsTrue(!tstValue, "WASHEDRBC-PLT");
            }



            // #4a Test CMV when no results
            //bloodUnitGuid1 = Guid.NewGuid();
            //drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E5464'");
            //unit = DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, (Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
            tstValue = PLTUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(!tstValue, "CMVNEG-PLT-NOResults*");

            // #4b Test CMV when a positive result
            DataCreator.CreateBloodUnitTest(Guid.Empty, PLTUnit.BloodUnitGuid, 103, "P");	// CMV Negative
            tstValue = PLTUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(!tstValue, "CMVNEG-PLT-WithResults");





            ////////////////////////////////////////////////////////
            ///OTHER		OTHER		OTHER		
          

            // #1a Irradiated, no C2 attribute
            bpGuid = DataCreator.GetRecordGuid("SELECT BloodProductGuid FROM BloodProduct WHERE ProductCode = 'E3805'");
            BOL.BloodUnit OTHERUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, bpGuid, false);

            OTHERUnit.ProductType.ProductTypeCode = "E040";
            OTHERUnit.ProductCode = "E3805";
            OTHERUnit.ProductAttributeForm = "@95 A2";
            tstValue = OTHERUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(!tstValue, "IRRADIATE-OTHER-NoC2");

            // #1b Irradiated, false C2
            OTHERUnit.ProductType.ProductTypeCode = "E040";
            OTHERUnit.ProductCode = "E3805";
            OTHERUnit.ProductAttributeForm = "@C2";
            tstValue = OTHERUnit.IsComponentRequirementSatisfied(COMMON.ComponentRequirement.IRRADIATE);
            Assert.IsTrue(!tstValue, "IRRADIATE-OTHER-WithC2");



            // #2a Test CMV when no results
            //bloodUnitGuid1 = Guid.NewGuid();
            //drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E3805'");
            //unit = DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, (Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
            tstValue = OTHERUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(!tstValue, "CMVNEG-OTHER-NoResults");

            // #2b Test CMV when a positive result
            DataCreator.CreateBloodUnitTest(Guid.Empty, OTHERUnit.BloodUnitGuid, 103, "P");	// CMV Negative
            tstValue = OTHERUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG);
            Assert.IsTrue(!tstValue, "CMVNEG-OTHER-WithResults");
        }


		[Test]
		public void GetBloodUnitsElgibleForCrossover_Pass()
		{
			DataTable dtTest = BOL.BloodUnit.GetBloodUnitsEligibleForCrossover();
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions)]
		public void GetBloodUnitsElgibleForCrossover_Fail()
		{
		}

		[Test]
		public void Save_Pass()
		{
			this.RefreshData = true;

			//Save Codabar
			this._tstCodabar.RecordStatusCode = COMMON.RecordStatusCode.Active;
			this._tstCodabar.BloodUnitStatus.UnitStatusDate = DateTime.Now;
			this._tstCodabar.BloodUnitStatus.LastUpdateUser = COMMON.LogonUser.LogonUserName;
			this._tstCodabar.BloodUnitStatus.QuarantineCannedCommentGuid = this._cannedCommentGuid;
			this._tstCodabar.BloodUnitStatus.DiscardCannedCommentGuid = this._cannedCommentGuid;

			this._tstCodabar.BloodUnitMedia.Comments = DateTime.Now.ToString();

			bool retValue = this._tstCodabar.Save(COMMON.UpdateFunction.UnitTests);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Save_Fail()
		{
			//Can't be failed because we can't override the RowVersions
		}

		
		[Test]
		public void SaveOutgoingShipment_Pass()
		{
			this.RefreshData = true;

			this._tstCodabar.RecordStatusCode = COMMON.RecordStatusCode.Active;
			this._tstCodabar.BloodUnitStatus.UnitStatusDate = DateTime.Now;
			this._tstCodabar.BloodUnitStatus.LastUpdateUser = COMMON.LogonUser.LogonUserName;
			this._tstCodabar.BloodUnitMedia.Comments = DateTime.Now.ToString();
			this._tstCodabar.BloodUnitStatus.QuarantineCannedCommentGuid = this._cannedCommentGuid;
			this._tstCodabar.BloodUnitStatus.DiscardCannedCommentGuid = this._cannedCommentGuid;

			BOL.OutgoingShipmentCollection tmpOSC = new BOL.OutgoingShipmentCollection(this._tstCodabar.IncomingShipmentGuid);

			//UPdate the pre-existing rows -- SaveOutgoing doesn't do that -- should it??
			UnitTestUtil.RunSQL("UPDATE BloodUnitShipment SET RecordStatusCode = 'X' WHERE OutgoingShipmentGuid IS NOT NULL AND BloodUnitGuid = '" + this._tstCodabar.BloodUnitGuid + "'");
			bool retValue = this._tstCodabar.SaveOutgoingShipment(tmpOSC, COMMON.UpdateFunction.UnitTests);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(Common.DataAccessLayerException))]
		public void SaveOutgoingShipment_Fail()
		{
			BOL.BloodUnit tmpBloodUnit = new BOL.BloodUnit(Guid.NewGuid());

			BOL.OutgoingShipmentCollection tmpOSC = new BOL.OutgoingShipmentCollection(Guid.NewGuid());

			bool retValue = tmpBloodUnit.SaveOutgoingShipment(tmpOSC, COMMON.UpdateFunction.UnitTests);
		}

		[Test]
		public void GenerateNumericCodabarUnitID_Pass()
		{
			bool boolVal = false;
			this._tstCodabar.LocalSupplier = new BOL.LocalSupplier(this._dtLocalSupplier.Rows[0][TABLES.LocalSupplier.FdaRegistrationNumber].ToString(), true);

			string tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12F9999999V", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length");

			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12FGKL8ppppp", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for FG");

			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12GFKL8ppppp", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for GF");

			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12KFGL8ppppp", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for KF");

			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12LFGK8ppppp", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for LF");

			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12FR9999992222", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for FR");

			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12GR99999922222", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for GR");

			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12KR333332222222", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for KR");

			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12LR333333322222", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for LR");
			try
			{
				tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12A333333322222", ref boolVal);
			}
			catch(BOL.BusinessObjectException)
			{
				Assert.IsTrue(true, "Passed test length for A");
			}

			
			tstValue = this._tstCodabar.GenerateNumericCodabarUnitID("12P99993322222", ref boolVal);
			Assert.AreEqual(true, tstValue.Length >= 0, "Length for P");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GenerateNumericCodabarUnitID_Fail()
		{
		}

		[Test]
		public void GetRbcAntigens_Pass()
		{
			DataTable dtTest = this._tstCodabar.GetRbcAntigens();
			
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetRbcAntigens_Fail()
		{
		}

		[Test]
		public void ContainsRedBloodCells_Pass()
		{
			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.WholeBlood);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "WholeBlood");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.RedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "RedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.WashedRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "WashedRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.FrozenRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "FrozenRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.FrozenRejuvenatedRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "FrozenRejuvenatedRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.DeglycerolizedRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "DeglycerolizedRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.DeglycerolizedRejuvenatedRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "DeglycerolizedRejuvenatedRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.RejuvenatedRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "RejuvenatedRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "ApheresisRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.WashedApheresisRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "WashedApheresisRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.FrozenApheresisRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "FrozenApheresisRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.DeglycerolizedApheresisRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "DeglycerolizedApheresisRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.RejuvenatedApheresisRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "RejuvenatedApheresisRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.FrozenRejuvenatedApheresisRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "FrozenRejuvenatedApheresisRedBloodCells");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.DeglycerolizedRejuvenatedApheresisRedBloodCells);
			Assert.AreEqual(true, this._tstCodabar.ContainsRedBloodCells(), "DeglycerolizedRejuvenatedApheresisRedBloodCells");
		}

		[Test]
		public void ContainsRedBloodCells_Fail()
		{
			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.FreshFrozenPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "FreshFrozenPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ThawedFreshFrozenPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ThawedFreshFrozenPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisFreshFrozenPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisFreshFrozenPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ThawedApheresisFreshFrozenPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ThawedApheresisFreshFrozenPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ThawedApheresisPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ThawedApheresisPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.LiquidPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "LiquidPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.Plasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "Plasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ThawedPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ThawedPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.PlateletRichPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "PlateletRichPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.Platelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "Platelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.WashedPlatelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "WashedPlatelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.PooledPlatelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "PooledPlatelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.WashedPooledPlatelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "WashedPooledPlatelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisPlatelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisPlatelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.FrozenApheresisPlatelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "FrozenApheresisPlatelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ThawedApheresisPlatelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ThawedApheresisPlatelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.WashedApheresisPlatelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "WashedApheresisPlatelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.Cryoprecipitate);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "Cryoprecipitate");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ThawedCryoprecipitate);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ThawedCryoprecipitate");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.PooledCryoprecipitate);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "PooledCryoprecipitate");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ThawedPooledCryoprecipitate);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ThawedPooledCryoprecipitate");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisCryoprecipitate);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisCryoprecipitate");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ThawedApheresisCryoprecipitate);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ThawedApheresisCryoprecipitate");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.Granulocytes);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "Granulocytes");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisGranulocytes);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisGranulocytes");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.PooledGranulocytes);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "PooledGranulocytes");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisGranulocytesPlatelets);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisGranulocytesPlatelets");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.Leukocytes);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "Leukocytes");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisLeukocytes);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisLeukocytes");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.PooledPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "PooledPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.LiquidApheresisPlasma);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "LiquidApheresisPlasma");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.PlateletRichBuffyCoat);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "PlateletRichBuffyCoat");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.PooledPlateletRichBuffyCoat);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "PooledPlateletRichBuffyCoat");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisLymphocytes);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisLymphocytes");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.ApheresisMonocytes);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "ApheresisMonocytes");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.Serum);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "Serum");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.PooledSerum);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "PooledSerum");

			this._tstCodabar.ProductType.ProductTypeCode = COMMON.Utility.GetProductTypeCodeFromProductTypeEnum(COMMON.ProductType.FrozenPooledSerum);
			Assert.AreEqual(false, this._tstCodabar.ContainsRedBloodCells(), "FrozenPooledSerum");
		}

		[Test]
		public void GenerateEyeReadableUnitID_Pass()
		{
			if (!this._tstCodabar.LocalSupplier.LabelUnitIDTrimIndicator)
				this._tstCodabar.LocalSupplier.LabelUnitIDTrimIndicator = true;

			string _unitID,_unitID2;
			long _longUnitID;
			int _intUnitID;
			string tstValue = this._tstCodabar.LabelUnitId;

			//Test with a LabelUnitIdTrimIndicator
			_longUnitID = System.Int64.Parse(tstValue);
			_unitID = tstValue.Substring(0,2);
			_intUnitID = Int16.Parse(_unitID)/20;
			_unitID = ((Int16.Parse(_unitID) % 20) + 1).ToString();
			_unitID2= (_intUnitID == 0) ? String.Empty : "FGKL".Substring(_intUnitID - 1,1);
			_unitID = "CEFGHJKLMNPQRSTVWXYZ".Substring(Int16.Parse(_unitID) - 1,1);
			tstValue = (_unitID2 + _unitID) + tstValue.Substring(2,5);

			//Test no prefix
			this._tstCodabar.LocalSupplier.FacilityPrefix = string.Empty;
			this._tstCodabar.GenerateEyeReadableUnitID();
			Assert.AreEqual(tstValue, this._tstCodabar.EyeReadableUnitId, "Codabar-true-noprefix");

			//Test with a prefix
			this._tstCodabar.LocalSupplier.FacilityPrefix = "12";
			this._tstCodabar.GenerateEyeReadableUnitID();
			Assert.AreEqual("12"+tstValue, this._tstCodabar.EyeReadableUnitId, "Codabar-true-prefix");

			//Test without a LabelUnitIdTrimIndicator
			this._tstCodabar.LocalSupplier.LabelUnitIDTrimIndicator = false;
			this._tstCodabar.GenerateEyeReadableUnitID();
			Assert.AreEqual(this._tstCodabar.LabelUnitId, this._tstCodabar.EyeReadableUnitId, "Codabar-false");


			//Test ISBT
			this._tstISBT.GenerateEyeReadableUnitID();
			Assert.AreEqual(this._tstISBT.LabelUnitId, this._tstISBT.EyeReadableUnitId, "ISBT");
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void GenerateEyeReadableUnitID_Fail()
		{
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(Guid.NewGuid());

			tmpBU.GenerateEyeReadableUnitID();
		}


		[Test]
		public void GenerateEyeReadableUnitID_LocalSupplier_String_Pass()
		{
			DataTable dtSupplier = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 LocalSupplierGuid FROM LocalSupplier WHERE LabelUnitIdTrimIndicator = 1 AND FacilityPrefix <> '' AND DivisionCode = '" + this.TestDivisionCode + "'", false); 
			BOL.LocalSupplier tmpSupplier = DataCreator.CreateLocalSupplier();
			tmpSupplier = new BOL.LocalSupplier(tmpSupplier.LocalSupplierGuid, true);
			tmpSupplier.LabelUnitIDTrimIndicator = true;
			tmpSupplier.UpdateLocalSupplier(Common.UpdateFunction.UnitTests);			

			Guid lsGuid = DataCreator.GetRecordGuid("SELECT TOP 1 LocalSupplierGuid FROM LocalSupplier WHERE LabelUnitIdTrimIndicator = 1 AND FacilityPrefix <> '' AND DivisionCode = '" + this.TestDivisionCode + "'"); 
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(lsGuid, false); 
			
			string _unitID,_unitID2;
			long _longUnitID;
			int _intUnitID;
			string tmpValue = this._tstCodabar.LabelUnitId;

			//Test with a LabelUnitIdTrimIndicator
			_longUnitID = System.Int64.Parse(tmpValue);
			_unitID = tmpValue.Substring(0,2);
			_intUnitID = Int16.Parse(_unitID)/20;
			_unitID = ((Int16.Parse(_unitID) % 20) + 1).ToString();
			_unitID2= (_intUnitID == 0) ? String.Empty : "FGKL".Substring(_intUnitID - 1,1);
			_unitID = "CEFGHJKLMNPQRSTVWXYZ".Substring(Int16.Parse(_unitID) - 1,1);
			tmpValue = (_unitID2 + _unitID) + tmpValue.Substring(2,5);

			//Test no prefix
			tmpLS.FacilityPrefix = string.Empty;
			string tstValue = BOL.BloodUnit.GenerateEyeReadableUnitID(tmpLS, this._tstCodabar.LabelUnitId);
			Assert.AreEqual(tmpValue, tstValue, "Codabar-noprefix");

			//Test with a prefix
			tmpLS.FacilityPrefix = "12";
			string tstValue2 = BOL.BloodUnit.GenerateEyeReadableUnitID(tmpLS, this._tstCodabar.LabelUnitId);
			Assert.AreEqual("12" + tmpValue, tstValue2, "Codabar-withprefix");
			
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void GenerateEyeReadableUnitID_LocalSupplier_String_Fail()
		{
			BOL.LocalSupplier tstLS = new BOL.LocalSupplier((Guid)this._dtLocalSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid], false);
			tstLS.LabelUnitIDTrimIndicator = true;

			BOL.BloodUnit.GenerateEyeReadableUnitID(tstLS, "BadUnitId");
		}

		[Test]
		public void GenerateEyeReadableUnitID_LocalSupplier_String_Fail2()
		{
			BOL.LocalSupplier tstLS = new BOL.LocalSupplier((Guid)this._dtLocalSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid], false);

			string tstValue = BOL.BloodUnit.GenerateEyeReadableUnitID(tstLS, string.Empty);

			Assert.AreEqual(string.Empty, tstValue, "Empty String");
		}

		[Test]
		public void GetBloodUnitByGuid_Pass()
		{
			DataTable dtTest = this._tstCodabar.GetBloodUnitByGuid(this._codabarGuid);
			Assert.AreEqual(true, dtTest.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		public void GetBloodUnitByGuid_Fail()
		{
			DataTable dtTest = this._tstCodabar.GetBloodUnitByGuid(Guid.NewGuid());
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}


		[Test]
		public void GetChildBloodUnitMessageString_Pass()
		{
			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE UnitStatusCode = 'M'", false);
			if (dtTemp.Rows.Count == 0)
			{
				//"M" Status units are Splits and Pools
				BOL.BloodUnit tmpNewUnit = DataCreator.CreateBloodUnit(true);

				BOL.BloodUnitModification tmpBUM = new BOL.BloodUnitModification(Common.ProductModification.SplitAdult, Common.ModifyUnitMethod.Closed, DateTime.Now.AddDays(-1), Common.LogonUser.LogonUserName);
				tmpBUM.SplitUnit(tmpNewUnit.BloodUnitGuid, 2);
				tmpBUM.Save(Common.UpdateFunction.UnitTests);
			}
			
			
			Guid parentGuid =  DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE UnitStatusCode = 'M'");

			BOL.BloodUnit tmpBU = new BOL.BloodUnit(parentGuid);
			string tstValue = tmpBU.GetChildBloodUnitMessageString(parentGuid);
			
			Assert.AreEqual(true, tstValue.Length > 0, "Length");
		}

		[Test]
		public void GetChildBloodUnitMessageString_Fail()
		{
			string tstValue = this._tstCodabar.GetChildBloodUnitMessageString(Guid.NewGuid());
			Assert.AreEqual(0, tstValue.Length, "Length");
		}

		[Test]
		public void GetPreviousBloodUnitStatusTextAndCodeByGuid_Pass()
		{
			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE UnitStatusCode = 'M'", false);
			if (dtTemp.Rows.Count == 0)
			{
				//"M" Status units are Splits and Pools
				BOL.BloodUnit tmpNewUnit = DataCreator.CreateBloodUnit(true);

				BOL.BloodUnitModification tmpBUM = new BOL.BloodUnitModification(Common.ProductModification.SplitAdult, Common.ModifyUnitMethod.Closed, DateTime.Now.AddDays(-1), Common.LogonUser.LogonUserName);
				tmpBUM.SplitUnit(tmpNewUnit.BloodUnitGuid, 2);
				tmpBUM.Save(Common.UpdateFunction.UnitTests);
			}

			Guid modGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE UnitStatusCode = 'M'");
			Object [] tstValue = this._tstCodabar.GetPreviousBloodUnitStatusTextAndCodeByGuid(modGuid);
			
			Assert.IsNotNull(tstValue, "Object");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetPreviousBloodUnitStatusTextAndCodeByGuid_Fail()
		{
		}

		[Test]
		public void RemoveFinalStatus_Pass()
		{
			this.RefreshData = true;

			Guid unitGuid = Guid.NewGuid();
			Guid patientGuid = (Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			Guid bloodProductGuid = (Guid)this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.BloodProductGuid];

			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), patientGuid, bloodProductGuid, true);
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid, bloodProductGuid);
			BOL.DiscardOrQuarantineBloodUnit dqBu = new BOL.DiscardOrQuarantineBloodUnit();
			dqBu.CommentGuid = (Guid)(BOL.CannedComment.GetCannedComments(this.TestDivisionCode, "QD").Rows[0][TABLES.CannedComment.CannedCommentGuid]);
			dqBu.CommentText = "Unit Test 4043";
			tmpBU.BloodUnitFinance.ReturnCreditAmount=(decimal)56.43;
			ArrayList arr = new ArrayList(1);
			arr.Add(tmpBU);
			dqBu.BloodUnits = arr;
			
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.DiscardUnit);
			dqBu.ChangeDiscardStatus(new ArrayList(),true,COMMON.UpdateFunction.UnitTests,COMMON.WorkloadProcessID.DiscardUnit);
			
			tmpBU = new BOL.BloodUnit(unitGuid, bloodProductGuid);
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.TransfuseUnit);
			bool tstValue = tmpBU.RemoveFinalStatus("Comments", COMMON.WorkloadProcessID.TransfuseUnit,null);
			Assert.AreEqual(true, tstValue, "Discard");

			//Code Coverage
			tmpBU = DataCreator.CreateBloodUnit(true);
			unitGuid = tmpBU.BloodUnitGuid;

			DataCreator.CreateOutgoingShipment(tmpBU.BloodUnitGuid, true);
			//Need to refresh the unit to update unit status
			tmpBU = new BOL.BloodUnit(unitGuid);
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogout);
			tstValue = tmpBU.RemoveFinalStatus("Comments", COMMON.WorkloadProcessID.UnitLogout,null);

			Assert.AreEqual(true, tstValue, "Transferred");
		}

		[Test]
		public void RemoveFinalStatus_OutgoingShipment_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			Guid patientGuid = (Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			Guid bloodProductGuid = (Guid)this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.BloodProductGuid];

			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), patientGuid, bloodProductGuid, true);
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			
			DataTable shipment=SprocHelper.GetInsertShipmentSprocTable(tmpBU.IncomingShipmentGuid , (Guid)this._dtLocalSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid], Guid.NewGuid());
			shipment.Columns.Add(TABLES.BloodUnitShipment.BloodUnitShipmentGuid, typeof(Guid));
			shipment.Columns.Add(TABLES.BloodUnitShipment.RowVersion, typeof(byte[]));
			shipment.Rows[0][TABLES.BloodUnitShipment.BloodUnitShipmentGuid]=(Guid)this._dtShipment.Rows[0][TABLES.BloodUnitShipment.BloodUnitShipmentGuid];
			shipment.Rows[0][TABLES.BloodUnitShipment.RowVersion]=(byte[])this._dtShipment.Rows[0][TABLES.BloodUnitShipment.RowVersion];

			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.TransfuseUnit);
			bool tstValue = tmpBU.RemoveFinalStatus("Comments", shipment, COMMON.WorkloadProcessID.TransfuseUnit, null);
			Assert.AreEqual(true, tstValue, "ReturnValue");

		}

		[Test]
		[ExpectedException(typeof(COMMON.RowVersionException))]
		public void RemoveFinalStatus_OutgoingShipment_Fail()
		{
			Guid unitGuid = Guid.NewGuid();
			Guid patientGuid = (Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			Guid bloodProductGuid = (Guid)this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.BloodProductGuid];

			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), patientGuid, bloodProductGuid, true);
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			
			DataTable shipment=SprocHelper.GetInsertShipmentSprocTable(tmpBU.IncomingShipmentGuid, (Guid)this._dtLocalSupplier.Rows[0][TABLES.LocalSupplier.LocalSupplierGuid], Guid.NewGuid());
			shipment.Columns.Add(TABLES.BloodUnitShipment.BloodUnitShipmentGuid, typeof(Guid));
			shipment.Columns.Add(TABLES.BloodUnitShipment.RowVersion, typeof(byte[]));
			shipment.Rows[0][TABLES.BloodUnitShipment.BloodUnitShipmentGuid]=(Guid)this._dtShipment.Rows[0][TABLES.BloodUnitShipment.BloodUnitShipmentGuid];
			shipment.Rows[0][TABLES.BloodUnitShipment.RowVersion]=new byte[8];

			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.TransfuseUnit);
			bool tstValue = tmpBU.RemoveFinalStatus("Comments", shipment, COMMON.WorkloadProcessID.TransfuseUnit, null);
			Assert.AreEqual(true, tstValue, "ReturnValue");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void RemoveFinalStatus_Fail()
		{
		}

		[Test]
		public void RemoveFinalStatus_PatientTransfusion_Pass()
		{
			Guid bloodUnitGuid = Guid.NewGuid();
			Guid patientGuid = (Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			Guid bloodProductGuid = (Guid)this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.BloodProductGuid];

			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), patientGuid, bloodProductGuid, true);

			//Insert an RBCx2 order, then transfuse 1 unit...
			Guid patientTreatmentGuid = Guid.NewGuid();
			this.InsertPatientTreatment(patientTreatmentGuid, patientGuid);

			Guid patientOrderGuid = Guid.NewGuid();
			this.InsertPatientOrder(patientOrderGuid, patientTreatmentGuid);

			Guid orderedComponentGuid = Guid.NewGuid();
			this.InsertOrderedComponent(orderedComponentGuid, patientOrderGuid, COMMON.ComponentClass.RBC, 2);

			Guid orderedUnitGuid = Guid.NewGuid();
			this.InsertOrderedUnit(orderedUnitGuid, orderedComponentGuid, bloodUnitGuid);

			Guid bloodUnitTestGuid1 = Guid.NewGuid();
			this.InsertBloodUnitTest(bloodUnitTestGuid1, bloodUnitGuid, orderedComponentGuid, orderedUnitGuid, 100, "0");

			Guid bloodUnitTestGuid2 = Guid.NewGuid();
			this.InsertBloodUnitTest(bloodUnitTestGuid2, bloodUnitGuid, orderedComponentGuid, orderedUnitGuid, 102, "C");

			Guid issuedUnitGuid = Guid.NewGuid();
			this.InsertIssuedUnit(issuedUnitGuid, bloodUnitGuid);

			Guid patientTransfusionGuid = Guid.NewGuid();
			this.InsertPatientTransfusion(patientTransfusionGuid, patientGuid, bloodUnitGuid);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(bloodUnitGuid);
			//
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.TransfuseUnit);
			bool tstValue = tmpBU.RemoveFinalStatus("Unit Test 3619", new BOL.PatientTransfusion(patientGuid, bloodUnitGuid, false), COMMON.WorkloadProcessID.TransfuseUnit, null);
			Assert.AreEqual(true, tstValue, "ReturnValue");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RemoveFinalStatus_PatientTransfusion_Fail()
		{			
			//BloodUnitStatus changes made failing this method impossible
		}

		[Test]
		public void RemoveFinalStatus_Modification_Pass()
		{
			BOL.BloodUnitModification tstBUM = new BOL.BloodUnitModification(COMMON.ProductModification.Pooled, COMMON.ModifyUnitMethod.Open, DateTime.Now, COMMON.LogonUser.LogonUserName);
			DataTable dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.Pooled, COMMON.ModifyUnitMethod.Open, false);
			if (dtUnits.Rows.Count == 0)
			{
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Pooled, Common.ModifyUnitMethod.Open, false);
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Pooled, Common.ModifyUnitMethod.Open, false);
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Pooled, Common.ModifyUnitMethod.Open, true);
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Pooled, Common.ModifyUnitMethod.Open, true);

				dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.Pooled, COMMON.ModifyUnitMethod.Open, true);
			}
			Guid bloodProductGuid = (Guid)dtUnits.Rows[0][TABLES.BloodUnit.BloodProductGuid];
			Guid bloodUnitGuid1 = Guid.NewGuid();
			Guid bloodUnitGuid2 = Guid.NewGuid();
			//Create units
			bool isCodabar = (dtUnits.Rows[0][TABLES.BloodUnit.LabelTypeCode].ToString() == "C");
			DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, bloodProductGuid, isCodabar);
			DataCreator.CreateBloodUnit(bloodUnitGuid2, Guid.NewGuid(), Guid.Empty, bloodProductGuid, isCodabar);

			//Modify units
			tstBUM.AddUnitToPool(new BOL.BloodUnit(bloodUnitGuid1, bloodProductGuid));
			tstBUM.AddUnitToPool(new BOL.BloodUnit(bloodUnitGuid2, bloodProductGuid));
			tstBUM.PoolUnits(this._lockManager);
			//
			if (!tstBUM.Save(COMMON.UpdateFunction.UnitTests)) Assert.AreEqual("Modification Data Setup - Failed", false);
			//
            this._lockManager.UnlockRecords();
			//Invalidate target unit
			DataTable dtBloodUnitModification = BOL.BloodUnitModification.GetBloodUnitModificationsByUnitGuid(bloodUnitGuid1);
			Guid targetGuid = (Guid)dtBloodUnitModification.Rows[0][TABLES.BloodUnitModification.ToBloodUnitGuid];
			BOL.BloodUnit targetUnit = new BOL.BloodUnit(targetGuid);
			COMMON.WorkloadProcessID workloadProcessID = COMMON.WorkloadProcessID.Unknown;
			BOL.BloodUnitInactivate.InactivateUnit(targetUnit, "Unit Test 7269", COMMON.UpdateFunction.UnitTests, new ArrayList(), ref workloadProcessID);
			
			//Grab the parent unit -- tstBUM.Save modified it's RowVersion
			BOL.BloodUnit parentUnit = new BOL.BloodUnit(bloodUnitGuid1);

			//Remove final status
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.PoolUnit);
			bool result = parentUnit.RemoveFinalStatus("Unit Test 7269", COMMON.WorkloadProcessID.PoolUnit, dtBloodUnitModification, new DataTable());
			//
			Assert.AreEqual(true, result, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(COMMON.RowVersionException))]
		public void RemoveFinalStatus_Modification_Fail()
		{
			BOL.BloodUnitModification tstBUM = new BOL.BloodUnitModification(COMMON.ProductModification.Pooled, COMMON.ModifyUnitMethod.Open, DateTime.Now, COMMON.LogonUser.LogonUserName);
			DataTable dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.Pooled, COMMON.ModifyUnitMethod.Open, false);
			if (dtUnits.Rows.Count == 0)
			{
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Pooled, Common.ModifyUnitMethod.Open, false);
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Pooled, Common.ModifyUnitMethod.Open, false);
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Pooled, Common.ModifyUnitMethod.Open, true);
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Pooled, Common.ModifyUnitMethod.Open, true);

				dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.Pooled, COMMON.ModifyUnitMethod.Open, true);
			}
			Guid bloodProductGuid = (Guid)dtUnits.Rows[0][TABLES.BloodUnit.BloodProductGuid];
			Guid bloodUnitGuid1 = Guid.NewGuid();
			Guid bloodUnitGuid2 = Guid.NewGuid();
			bool isCodabar = (dtUnits.Rows[0][TABLES.BloodUnit.LabelTypeCode].ToString() == "C");
			//Create units
			DataCreator.CreateBloodUnit(bloodUnitGuid1, Guid.NewGuid(), Guid.Empty, bloodProductGuid, isCodabar);
			DataCreator.CreateBloodUnit(bloodUnitGuid2, Guid.NewGuid(), Guid.Empty, bloodProductGuid, isCodabar);

			//Modify units
			tstBUM.AddUnitToPool(new BOL.BloodUnit(bloodUnitGuid1, bloodProductGuid));
			tstBUM.AddUnitToPool(new BOL.BloodUnit(bloodUnitGuid2, bloodProductGuid));
			tstBUM.PoolUnits(this._lockManager);
			//
			if (!tstBUM.Save(COMMON.UpdateFunction.UnitTests)) Assert.AreEqual("Modification Data Setup - Failed", false);
            this._lockManager.UnlockRecords();
			//
			//Invalidate target unit
			DataTable dtBloodUnitModification = BOL.BloodUnitModification.GetBloodUnitModificationsByUnitGuid(bloodUnitGuid1);
			Guid targetGuid = (Guid)dtBloodUnitModification.Rows[0][TABLES.BloodUnitModification.ToBloodUnitGuid];
			BOL.BloodUnit targetUnit = new BOL.BloodUnit(targetGuid);
			COMMON.WorkloadProcessID workloadProcessID = COMMON.WorkloadProcessID.Unknown;
			BOL.BloodUnitInactivate.InactivateUnit(targetUnit, "Unit Test 7270", COMMON.UpdateFunction.UnitTests, new ArrayList(), ref workloadProcessID);
			//
			//Change row version
			DataRow dr = dtBloodUnitModification.Rows[0];
			dr[TABLES.BloodUnitModification.RowVersion] = new byte[8];
			
			//Grab the parent unit -- tstBUM.Save modified it's RowVersion
			BOL.BloodUnit parentUnit = new BOL.BloodUnit(bloodUnitGuid1);

			//Remove final status
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.PoolUnit);
			bool result = parentUnit.RemoveFinalStatus("Unit Test 7270", COMMON.WorkloadProcessID.PoolUnit, dtBloodUnitModification, new DataTable());
			//
			Assert.AreEqual(true, result, "ReturnValue");
		}

		[Test]
		public void FreeDirectedBloodUnit_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			Guid statusGuid = Guid.NewGuid();
			Guid patientGuid = (Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			Guid bloodProductGuid = (Guid)this._dtBloodUnitCodabar.Rows[0][TABLES.BloodUnit.BloodProductGuid];

			DataCreator.CreateBloodUnit(unitGuid, statusGuid, patientGuid, bloodProductGuid, true);

			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid);

			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.ReleaseDirectedUnit);
			bool retValue = tmpUnit.FreeDirectedBloodUnit(COMMON.LogonUser.LogonUserName, DateTime.Now, "Comments", COMMON.WorkloadProcessID.ReleaseDirectedUnit);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void FreeDirectedBloodUnit_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			DataRow dr = this._tstCodabar.LoadDataRowFromThis(this._dtBloodUnitCodabar.NewRow());

			Assert.AreEqual(this._tstCodabar.BloodUnitGuid, (Guid)dr[TABLES.BloodUnit.BloodUnitGuid], "BloodUnitGuid");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}

		[Test]
		public void Save_Bool_Arraylist_Arraylist_Pass()
		{
			// First, try save with new antigen and blood unit test data (insert)
			#region Insert Test 1

			Guid unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._tstCodabar.BloodProductGuid, true);
			//
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.LastUpdateDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			tmpBU.BloodUnitMedia.IsDirty = true;
			tmpBU.BloodUnitStatus.IsDirty = true;
			//
			System.Collections.ArrayList antigenList = new System.Collections.ArrayList();
			System.Collections.ArrayList bloodUnitTestList = new System.Collections.ArrayList();
			//
			BOL.BloodUnitAntigen antigenDib = new BOL.BloodUnitAntigen();
			antigenDib.BloodUnitGuid = tmpBU.BloodUnitGuid;
			antigenDib.AntigenTypeId = 17;
			antigenDib.Positive = true;
			antigenDib.IsNew = true;
			antigenList.Add(antigenDib);
			//
			BOL.BloodUnitAntigen antigenAub = new BOL.BloodUnitAntigen();
			antigenAub.BloodUnitGuid = tmpBU.BloodUnitGuid;
			antigenAub.AntigenTypeId = 4;
			antigenAub.Positive = true;
			antigenAub.IsNew = true;
			antigenList.Add(antigenAub);
			//
			BOL.BloodUnitTest cmvTest = new BOL.BloodUnitTest( tmpBU.BloodUnitGuid, Convert.ToInt32( COMMON.TestType.CMV ) );
			cmvTest.EntryMethodCode = "M";
			cmvTest.IsNew = true;
			cmvTest.TestDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			cmvTest.TestResultId = "P";
			bloodUnitTestList.Add(cmvTest);
			//
			BOL.BloodUnitTest sickleCellTest = new BOL.BloodUnitTest( tmpBU.BloodUnitGuid, Convert.ToInt32( COMMON.TestType.SickleCell ) );
			sickleCellTest.EntryMethodCode = "M";
			sickleCellTest.IsNew = true;
			sickleCellTest.TestDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			sickleCellTest.TestResultId = "P";
			bloodUnitTestList.Add(sickleCellTest);
			//
			bool quaratineUnit = false;
			//
			Assert.AreEqual(true, tmpBU.Save(true, false, quaratineUnit, antigenList, bloodUnitTestList, COMMON.UpdateFunction.UnitTests), "Save1" );

			#endregion
			//
			// Okay, now try with existing antigen and blood unit test data (update)
			// We will quarantine the unit to test BR_41.12 implementation
			#region Update Test 1

			tmpBU = new BOL.BloodUnit(unitGuid);
			//
			antigenList.Clear();
			DataTable antigenTable = BOL.BloodUnitAntigen.GetBloodUnitAntigens( tmpBU.BloodUnitGuid, COMMON.LogonUser.LogonUserDivisionCode );
			Assert.AreEqual (2, antigenTable.Rows.Count, "Rows.Count");
			antigenDib = new BOL.BloodUnitAntigen(antigenTable.Rows[0]);
			antigenList.Add( antigenDib );
			//
			BOL.BloodUnitAntigen antigenDib2 = new BOL.BloodUnitAntigen(antigenTable.Rows[0]);
			antigenDib2.Positive = false;
			antigenList.Add( antigenDib2 );
			//
			antigenAub = new BOL.BloodUnitAntigen(antigenTable.Rows[1]);
			antigenAub.Positive = false;
			antigenAub.RecordStatusCode = COMMON.RecordStatusCode.Inactive;
			antigenList.Add( antigenAub );
			//
			bloodUnitTestList.Clear();
			DataTable bloodUnitTestTable = BOL.BloodUnitTest.GetBloodUnitTests( tmpBU.BloodUnitGuid, COMMON.LogonUser.LogonUserDivisionCode );
			int count = bloodUnitTestTable.Rows.Count;
			Assert.AreEqual (2, bloodUnitTestTable.Rows.Count, "Rows.Count");
			//
			cmvTest = new BOL.BloodUnitTest(  bloodUnitTestTable.Rows[0] );
			cmvTest.TestResultId = "N";
			bloodUnitTestList.Add( cmvTest );
			//
			sickleCellTest = new BOL.BloodUnitTest(  bloodUnitTestTable.Rows[1] );
			sickleCellTest.TestResultId = "N";
			bloodUnitTestList.Add( sickleCellTest );
			//
			quaratineUnit = true;
			//
			Assert.AreEqual(true, tmpBU.Save(true, false, quaratineUnit, antigenList, bloodUnitTestList, COMMON.UpdateFunction.UnitTests),  "Save2" );
			//
			tmpBU = new BOL.BloodUnit(unitGuid);
			//
			Assert.IsTrue(tmpBU.BloodUnitStatus.QuarantineIndicator == true);

			#endregion
			//
			// Delete test blood unit data
			Assert.AreEqual(true, DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString()),  "Clear2" );
			//
			#region Insert Test 2 : different property values

			// Need a patient guid
			Guid patientGuid = DataCreator.GetRandomRecordGuid(TABLES.Patient.TableName,"");

			unitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, this._tstCodabar.BloodProductGuid, true);
			//
			tmpBU = new BOL.BloodUnit(unitGuid);
			tmpBU.LastUpdateDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			tmpBU.DiscardedPlasmaVolume = 75;
			tmpBU.BloodUnitMedia.IsDirty = true;
			tmpBU.BloodUnitMedia.TestingStatusId = 1;
			tmpBU.BloodUnitMedia.DirectedPatientGuid = patientGuid;
			tmpBU.BloodUnitStatus.IsDirty = true;
			tmpBU.BloodUnitStatus.BiohazardousWarningIndicator = true;
			//
			quaratineUnit = false;
			//
			Assert.AreEqual(true, tmpBU.Save(true, false, quaratineUnit, new ArrayList(), new ArrayList(), COMMON.UpdateFunction.UnitTests), "Save3" );

			#endregion
			//
			// Delete test blood unit data
			Assert.AreEqual(true, DataKiller.ClearBloodUnitData(tmpBU.BloodUnitGuid.ToString()), "Clear3" );
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void Save_Bool_Arraylist_Arraylist_Fail()
		{
			BOL.BloodUnit tmpBU = new BOL.BloodUnit(Guid.NewGuid());
			tmpBU.Save(true, false, false, new ArrayList(), new ArrayList(), COMMON.UpdateFunction.UnitTests); 
		}

		[Test]
		public void SetLabelUnitId_Pass()
		{
			string unitId = "1234";

			this._tstCodabar.SetLabelUnitId( unitId, false );

			Assert.IsTrue( this._tstCodabar.LabelUnitId.Equals( unitId ) );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SetLabelUnitId_Fail()
		{
		}

		[Test]
		public void SetEyeReadableUnitId_Pass()
		{
			string unitId = "1234";

			this._tstCodabar.SetEyeReadableUnitId( unitId, false );

			Assert.IsTrue( this._tstCodabar.EyeReadableUnitId.Equals( unitId ) );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SetEyeReadableUnitId_Fail()
		{
		}

		[Test]
		public void GetIncomingShipmentBloodUnitsString_Pass()
		{
			string query = string.Concat("SELECT TOP 1 ShipmentGuid FROM Shipment ", 
										"JOIN BloodUnitShipment ON Shipment.ShipmentGuid = BloodUnitShipment.IncomingShipmentGuid ",
										"WHERE BloodUnitShipment.RecordStatusCode = 'A' AND Shipment.RecordStatusCode = 'A' AND Shipment.DivisionCode = '"+UnitTestConfig.TestDivisionCode+"'");
			//
			Guid shipmentGuid = DataCreator.GetRecordGuid( query );
			//
			Assert.IsTrue( BOL.BloodUnit.GetIncomingShipmentBloodUnitsString( shipmentGuid ).Count > 0 );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetIncomingShipmentBloodUnitsString_Fail()
		{
		}

		[Test]
		public void RemoveUnitModification_Pass()
		{
			BOL.BloodUnitModification tstBUM = new BOL.BloodUnitModification(COMMON.ProductModification.Irradiated, COMMON.ModifyUnitMethod.Closed, DateTime.Now, COMMON.LogonUser.LogonUserName);
			DataTable dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.Irradiated, COMMON.ModifyUnitMethod.Closed, false);
			if (dtUnits.Rows.Count == 0)
			{
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Irradiated, Common.ModifyUnitMethod.Closed, false);
				dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.Irradiated, COMMON.ModifyUnitMethod.Closed, false);
			}
			Guid bloodProductGuid = (Guid)dtUnits.Rows[0][TABLES.BloodUnit.BloodProductGuid];
			Guid bloodUnitGuid = Guid.NewGuid();
			//Create unit
			bool isCodabar = (dtUnits.Rows[0][TABLES.BloodUnit.LabelTypeCode].ToString() == "C");
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), Guid.Empty, bloodProductGuid, isCodabar);

			//Irradiate unit
			tstBUM.ModifyUnit(bloodUnitGuid);
			DataTable dtTargets = tstBUM.GetTargetBloodProducts(tstBUM.GetBloodProductGuidArray());
			tstBUM.ModifiedUnit.TargetBloodProduct = new BOL.BloodProduct(dtTargets.Rows[0]);

			//Save the Modification
			if (!tstBUM.Save(COMMON.UpdateFunction.UnitTests)) Assert.AreEqual("Modification Data Setup - Failed", false);
			//
			//Grab the parent unit -- tstBUM.Save modified it's RowVersion
			BOL.BloodUnit parentUnit = new BOL.BloodUnit(bloodUnitGuid);

			//Remove final status
			COMMON.WorkloadProcessID workloadProcessId = COMMON.WorkloadProcessID.Unknown;
			bool result = parentUnit.RemoveUnitModification(BOL.BloodUnitModification.GetBloodUnitModificationsByUnitGuid(bloodUnitGuid).Rows[0], "Unit Test 7385", ref workloadProcessId,null);
			//
			Assert.AreEqual(true, result, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(COMMON.RowVersionException))]
		public void RemoveUnitModification_Fail()
		{
			BOL.BloodUnitModification tstBUM = new BOL.BloodUnitModification(COMMON.ProductModification.Irradiated, COMMON.ModifyUnitMethod.Closed, DateTime.Now, COMMON.LogonUser.LogonUserName);
			
			DataTable dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.Irradiated, COMMON.ModifyUnitMethod.Closed,false);
			if (dtUnits.Rows.Count == 0)
			{
				Guid newUnitGuid = Guid.NewGuid();
				DataCreator.CreateModifiableBloodUnit(Common.ProductModification.Irradiated, Common.ModifyUnitMethod.Closed, false);
				dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.Irradiated, COMMON.ModifyUnitMethod.Closed, false);
			}

			Guid bloodProductGuid = (Guid)dtUnits.Rows[0][TABLES.BloodUnit.BloodProductGuid];
			Guid bloodUnitGuid = Guid.NewGuid();
			//Create unit
			bool isCodabar = (dtUnits.Rows[0][TABLES.BloodUnit.LabelTypeCode].ToString() == "C");
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), Guid.Empty, bloodProductGuid, isCodabar);

			//Irradiate unit
			tstBUM.ModifyUnit(bloodUnitGuid);
			DataTable dtTargets = tstBUM.GetTargetBloodProducts(tstBUM.GetBloodProductGuidArray());
			tstBUM.ModifiedUnit.TargetBloodProduct = new BOL.BloodProduct(dtTargets.Rows[0]);

			//Save the Modification
			if (!tstBUM.Save(COMMON.UpdateFunction.UnitTests)) Assert.AreEqual("Modification Data Setup - Failed", false);
			
			//Grab the parent unit -- tstBUM.Save modified it's RowVersion
			BOL.BloodUnit parentUnit = new BOL.BloodUnit(bloodUnitGuid);

			//Change row version
			parentUnit.RowVersion = new byte[8];

			//Remove final status
			COMMON.WorkloadProcessID workloadProcessId = COMMON.WorkloadProcessID.Unknown;
			bool result = parentUnit.RemoveUnitModification(BOL.BloodUnitModification.GetBloodUnitModificationsByUnitGuid(bloodUnitGuid).Rows[0], "Unit Test 7386", ref workloadProcessId,null );
		}

		[Test]
		public void RemoveUnitModification_Fail_Pool()
		{
			DataTable dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.EditPool, COMMON.ModifyUnitMethod.Closed, false);
			if (dtUnits.Rows.Count == 0)
			{
				new UnitTests.BOL_BloodUnitModification().CreatePoolForTests(false, false);
				dtUnits = BOL.BloodUnitModification.GetModifiableBloodUnits(COMMON.ProductModification.EditPool, COMMON.ModifyUnitMethod.Closed, true);
			}
			BOL.BloodUnit bu = new BOL.BloodUnit(dtUnits.Rows[0]);
			//
			DataRow drMod = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM BloodUnitModification WHERE ToBloodUnitGuid = '" + bu.BloodUnitGuid.ToString() + "'");
			//
			COMMON.WorkloadProcessID workloadProcessId = COMMON.WorkloadProcessID.Unknown;
			bool result = this._tstCodabar.RemoveUnitModification(drMod, "Unit Test 8138", ref workloadProcessId,null);
			//
			Assert.AreEqual(false, result, "ModificationCode");
		}

		[Test]
		public void GetModifiedBloodUnits_Pass()
		{
			int cntr = BOL.BloodUnit.GetModifiedBloodUnits().Rows.Count;

			if (cntr == 0)
			{
				//Split, Closed is the easiest modify there is, and naturally, this method ignores splits and pools
				BOL.BloodUnit tmpUnit = DataCreator.CreateModifiableBloodUnit(COMMON.ProductModification.Irradiated, COMMON.ModifyUnitMethod.Closed, false);
				BOL.BloodUnitModification bum = new BOL.BloodUnitModification(COMMON.ProductModification.Irradiated, COMMON.ModifyUnitMethod.Closed, DateTime.Now.AddDays(-1), COMMON.LogonUser.LogonUserName);
				bum.ModifyUnit(tmpUnit.BloodUnitGuid);
				DataTable dtTargets = bum.GetTargetBloodProducts(bum.GetBloodProductGuidArray());
				bum.ModifiedUnit.TargetBloodProduct = new BOL.BloodProduct(dtTargets.Rows[0]);
				bum.Comments = "UnitTest7983";
				bum.Save(COMMON.UpdateFunction.UnitTests);

				cntr = BOL.BloodUnit.GetModifiedBloodUnits().Rows.Count;
			}
			if(cntr == 0)
			{
				Assert.AreEqual(false, (cntr == 0), "No Modified Units");
			}
			else
			{
				Assert.IsTrue(cntr > 0);
			}
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetModifiedBloodUnits_Fail()
		{
		}

		#region Helper Methods

		private void SetupDirectedUnitData()
		{
			// Set up 1 blood unit that matches the req'd conditions
			System.Text.StringBuilder query = new System.Text.StringBuilder("UPDATE BloodUnit ");
			query.Append("SET DonationTypeId = 12 WHERE BloodUnitGuid IN (");
			query.Append("SELECT TOP 1(BloodUnitGuid) FROM vw_BloodShipAllCurrent as vwBLWCS ");
			query.Append("WHERE vwBLWCS.DivisionCode = '").Append(UnitTestConfig.TestDivisionCode).Append("' ");
			query.Append("AND (vwBLWCS.UnitStatusCode NOT IN ('D', 'M', 'T', 'X')) ");
			query.Append("AND (vwBLWCS.DirectedPatientGuid IS NOT NULL)");
			query.Append("AND (SELECT COUNT(1) from dbo.Patient ");
			query.Append("JOIN dbo.PatientTreatment ON dbo.Patient.PatientGuid = PatientTreatment.PatientGuid ");
			query.Append("JOIN dbo.PatientOrder ON dbo.PatientTreatment.PatientTreatmentGuid = dbo.PatientOrder.PatientTreatmentGuid ");
			query.Append("JOIN dbo.OrderedComponent ON dbo.PatientOrder.PatientOrderGuid = dbo.OrderedComponent.PatientOrderGuid ");
			query.Append("JOIN dbo.OrderedUnit ON dbo.OrderedComponent.OrderedComponentGuid = dbo.OrderedUnit.OrderedComponentGuid ");
			query.Append("AND dbo.OrderedUnit.BloodUnitGuid = vwBLWCS.BloodUnitGuid) = 0 )");

			UnitTestUtil.RunSQL(query.ToString());
		}

		private Guid GetUnitInFinalStatusWithValidHistory()
		{
			DataRow dr = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE RecordStatusCode = 'A' AND UnitStatusCode = 'M' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			Guid unitGuid = (Guid)dr[TABLES.BloodUnit.BloodUnitGuid];

			return unitGuid;
		}

		private void InsertPatientTreatment(Guid patientTreatmentGuid, Guid patientGuid)
		{
			string insertPatientTreatment = "INSERT INTO PATIENTTREATMENT (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Environment.UserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(COMMON.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertPatientTreatment);
		}

		private void InsertPatientOrder(Guid patientOrderGuid, Guid patientTreatmentGuid)
		{
			string insertPatientOrder = "INSERT INTO PATIENTORDER (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + COMMON.LogonUser.LogonUserDivisionCode + "',";
			insertPatientOrder += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Environment.UserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(COMMON.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertPatientOrder);
		}

		private void InsertOrderedComponent(Guid orderedComponentGuid, Guid patientOrderGuid, COMMON.ComponentClass componentClass, int quantity)
		{
			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(componentClass).ToString() + "',";
            insertOrderedComponent += "'" + SprocHelper.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += quantity.ToString() + ",";
			insertOrderedComponent += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + COMMON.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedComponent += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Environment.UserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(COMMON.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertOrderedComponent);
		}

		private void InsertOrderedUnit(Guid orderedUnitGuid, Guid orderedComponentGuid, Guid bloodUnitGuid)
		{
			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + COMMON.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedUnit += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(COMMON.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertOrderedUnit);
		}

		private void InsertBloodUnitTest(Guid bloodUnitTestGuid, Guid bloodUnitGuid, Guid orderedComponentGuid, Guid orderedUnitGuid, int bloodTestTypeId, string testResultId)
		{
			string insertBloodUnitTest = "INSERT INTO BLOODUNITTEST (";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodUnitTestGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodUnitGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodTestTypeId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestDate + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestTechId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestResultId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.EntryTechId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.EntryMethodCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.OrderedComponentGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.OrderedUnitGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.ReactivityPhaseCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.RecordStatusCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.DivisionCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateDate + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateUser + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateFunctionId + ")";
			insertBloodUnitTest += "VALUES (";
			insertBloodUnitTest += "'" + bloodUnitTestGuid.ToString() + "',";
			insertBloodUnitTest += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitTest += bloodTestTypeId.ToString() + ",";
			insertBloodUnitTest += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'" + testResultId + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'M',";
			insertBloodUnitTest += "'" + orderedComponentGuid.ToString() + "',";
			insertBloodUnitTest += "'" + orderedUnitGuid.ToString() + "',";
			insertBloodUnitTest += "'2',";
			insertBloodUnitTest += "'A',";
			insertBloodUnitTest += "'" + COMMON.LogonUser.LogonUserDivisionCode + "',";
			insertBloodUnitTest += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'" + Convert.ToInt32(COMMON.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertBloodUnitTest);
		}

		private void InsertIssuedUnit(Guid issuedUnitGuid, Guid bloodUnitGuid)
		{
			string insertIssuedUnit = "INSERT INTO ISSUEDUNIT (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(COMMON.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertIssuedUnit);
		}

		private void InsertPatientTransfusion(Guid patientTransfusionGuid, Guid patientGuid, Guid bloodUnitGuid)
		{
			string insertPatientTransfusion = "INSERT INTO PATIENTTRANSFUSION (";
			insertPatientTransfusion += TABLES.PatientTransfusion.PatientTransfusionGuid + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.PatientGuid + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.BloodUnitGuid + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.ReactionIndicator + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.OtherDifficultyIndicator + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.DivisionCode + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.LastUpdateDate + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.LastUpdateUser + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.LastUpdateFunctionId + ")";
			insertPatientTransfusion += "VALUES (";
			insertPatientTransfusion += "'" + patientTransfusionGuid.ToString() + "',";
			insertPatientTransfusion += "'" + patientGuid.ToString() + "',";
			insertPatientTransfusion += "'" + bloodUnitGuid.ToString() + "',";
			insertPatientTransfusion += "0,";
			insertPatientTransfusion += "0,";
			insertPatientTransfusion += "'" + COMMON.LogonUser.LogonUserDivisionCode + "',";
			insertPatientTransfusion += "'" + COMMON.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTransfusion += "'" + Environment.UserName + "',";
			insertPatientTransfusion += "'" + Convert.ToInt32(COMMON.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertPatientTransfusion);
		}

		#endregion
	}
}
#endif
